#include "all03.h"

#define _VERSION_			"1.04"

//----------------------------
#define PIC_LOAD_CFG		0x00		// (16<-) switch to 0x2000 (0x4000)
#define PIC_INCREMENT		0x06		// next address

#define PIC_LOAD_CODE		0x02		// (16<-)
#define PIC_READ_CODE		0x04		// (->16)

#define PIC_START_PROG		0x08		// some devices ERASE & PROGRAM !

#define PIC_PROG_ONLY		0x18		//
#define PIC_END_PROG		0x0E		//

// Devices with ERASE function
//----------------------------
#define PIC_CHIP_ERASE		0x1F		// Chip Erase TYPE=1

#define PIC_CODE_ERASE		0x09		// Single mem Erase TYPE=1

#define PIC_BULK_SETUP1		0x01		// Erase Setup TYPE=0
#define PIC_BULK_SETUP2		0x07		// ChipErase if ADDR=2007

// Flash chips only
//----------------------------
#define PIC_LOAD_DATA		0x03		// (16<-)
#define PIC_READ_DATA		0x05		// (->16)

#define PIC_DATA_ERASE		0x0B		// works only if not protected

// Segment names
//----------------------------
#define CFG_SEG				PIC_LOAD_CFG
#define DATA_SEG			PIC_LOAD_DATA
#define CODE_SEG			PIC_LOAD_CODE

//----------------------------
int _MCLR, _DATA, _CLK, _PGM, _LVP;
int _VCC1, _VCC2, _GND1, _GND2;
int _UB, _VPP;

struct DEV {
	char name[20];
	unsigned Size, DataSize;	// ROM & EEPROM sizes in bytes(!)
	char mclr,dat,clk, lvp;		// MCLR, DATA, CLOCK and LVP pin numbers (*)
	char vcc1,vcc2, gnd1,gnd2;	// Pins numbers for VCC(s) and GND(s)    (*)
	int UB, UBmin, UBmax;		// Norm, min. & max. Vcc
	int VPP, VPP1st;			// VPP value, apply Vpp **BEFORE** Vcc
	char erase_type;			// if electrically erasable, type of erase
	char NumProgBytes;			// Number of words to load for 1 prog cycle
	char prog_type;				// prog pulse type
	unsigned cfg_mask;			// implemented bits in CONFIG word
};
								// (*) Pin numbers on 40pin socket !!!
// erase types =
// 0 : device has no erase function
// 1 : LOAD_XXX with 3FFF, BULK_SETUP, START_PROG, BULK_SETUP
// 2 : XXX_ERASE, START_PROG
// 3 : LOAD_XXX with 3FFF, XXX_ERASE, START_PROG (all cmds with appended loads)

// prog types =
// 0 : max. 25 * (START_PROG / wait 100us / END_PROG) + 3*n postpulses
// 1 : 1 PROG_ONLY pulse, then 4ms delay
// 2 : 1 START_PROG pulse, then 8ms delay
// 3 : 1 PROG_ONLY pulse, then 20ms delay, END_PROG

struct DEV PIC_devs[] = {
//			TYPE		ROMSIZE	EESIZE mclr	dat	clk	lvp vcc vcc gnd	gnd	Ub  UbL	UbH Vpp 1st ERA	NUM	PRG MASK
/* 0*/	{ "PIC16C554",	0x0400,	0x0000, 15,	24,	23,	0,	25,	0,	16,	27,	50,	20,	65, 120,0,  0,	1,	0,	0x3F3F },
/* 1*/	{ "PIC16C557",	0x1000,	0x0000, 34,	31,	30,	0,	8,	0,	10,	33,	50,	20,	65,	120,0,  0,	1,	0,	0x3F3F },
/* 2*/	{ "PIC16C558",	0x1000,	0x0000, 15,	24,	23,	0,	25,	0,	16,	27,	50,	20,	65,	120,0,  0,	1,	0,	0x3F3F },
/* 3*/	{ "PIC16C710",	0x0800,	0x0000, 15,	24,	23,	0,	25,	0,	16,	0,	50,	25,	55,	120,0,  0,	1,	0,	0x001F },
/* 4*/	{ "PIC16C71",	0x1000,	0x0000, 15,	24,	23,	0,	25,	0,	16,	0,	50,	25,	55,	120,0,  0,	1,	0,	0x001F },
/* 5*/	{ "PIC16C711",	0x1000,	0x0000, 15,	24,	23,	0,	25,	0,	16,	0,	50,	25,	55,	120,0,  0,	1,	0,	0x001F },
/* 6*/	{ "PIC16C715",	0x2000,	0x0000, 15,	24,	23,	0,	25,	0,	16,	0,	50,	25,	55,	120,0,  0,	1,	0,	0x001F },
/* 7*/	{ "PIC16F870",	0x1000, 0x0040, 7,	34,	33,	0,	26,	0,	14,	25,	50,	25,	55,	120,0,  1,	1,	1,	0x3BFF },
/* 8*/	{ "PIC16F871",	0x1000, 0x0040, 1,	40,	39,	0,	11,	32,	12,	31,	50,	25,	55,	120,0,  1,	1,	1,	0x3BFF },
/* 9*/	{ "PIC16F872",	0x1000, 0x0040, 7,	34,	33,	0,	26,	0,	14,	25,	50,	25,	55,	120,0,  1,	1,	1,	0x3BFF },
/*10*/	{ "PIC16F873",	0x2000, 0x0080, 7,	34,	33,	0,	26,	0,	14,	25,	50,	25,	55,	120,0,  1,	1,	1,	0x3BFF },
/*11*/	{ "PIC16F873A",	0x2000, 0x0080, 7,	34,	33,	0,	26,	0,	14,	25,	50,	20,	55,	120,0,  2,	8,	2,	0x3BFF },
/*12*/	{ "PIC16F874",	0x2000, 0x0080, 1,	40,	39,	0,	11,	32,	12,	31,	50,	25,	55,	120,0,  1,	1,	1,	0x3BFF },
/*13*/	{ "PIC16F874A",	0x2000, 0x0080, 1,	40,	39,	0,	11,	32,	12,	31,	50,	20,	55,	120,0,  2,	8,	2,	0x3BFF },
/*14*/	{ "PIC16F876",	0x4000, 0x0100, 7,	34,	33,	0,	26,	0,	14,	25,	50,	25,	55,	120,0,  1,	1,	1,	0x3BFF },
/*15*/	{ "PIC16F876A",	0x4000, 0x0100, 7,	34,	33,	0,	26,	0,	14,	25,	50,	20,	55,	120,0,  2,	8,	2,	0x3BFF },
/*16*/	{ "PIC16F877",	0x4000, 0x0100, 1,	40,	39,	0,	11,	32,	12,	31,	50,	25,	55,	120,0,  1,	1,	1,	0x3BFF },
/*17*/	{ "PIC16F877A",	0x4000, 0x0100, 1,	40,	39,	0,	11,	32,	12,	31,	50,	20,	55,	120,0,  2,	8,	2,	0x3BFF },
/*18*/	{ "PIC16F83",	0x0400, 0x0040, 15,	24,	23,	0,	25,	0,	16,	27,	50,	20,	60,	120,0,  1,	1,	3,	0x3FFF },
/*19*/	{ "PIC16F84",	0x0800, 0x0040, 15,	24,	23,	0,	25,	0,	16,	27,	50,	20,	60,	120,0,  1,	1,	3,	0x3FFF },
/*20*/	{ "PIC16F84A",	0x0800, 0x0040, 15,	24,	23,	0,	25,	0,	16,	27,	50,	20,	60,	120,0,  1,	1,	2,	0x3FFF },
/*21*/	{ "PIC16F627",	0x0800, 0x0080, 15,	24,	23,	21,	25,	0,	16,	27,	50,	30, 55, 120,1,  3,  1,  2,	0x3DFF },
/*22*/	{ "PIC16F627A",	0x0800, 0x0080, 15,	24,	23,	21,	25,	0,	16,	27,	50,	30, 55, 120,1,  2,  1,  2,	0x21FF },
/*23*/	{ "PIC16F628",	0x1000, 0x0080, 15,	24,	23,	21,	25,	0,	16,	27,	50,	30, 55, 120,1,  3,  1,  2,	0x3DFF },
/*24*/	{ "PIC16F628A",	0x1000, 0x0080, 15,	24,	23,	21,	25,	0,	16,	27,	50,	30, 55, 120,1,  2,  1,  2,	0x21FF },
/*25*/	{ "PIC16F648A",	0x2000, 0x0100, 15,	24,	23,	21,	25,	0,	16,	27,	50,	30, 55, 120,1,  2,  1,  2,	0x21FF }
};

struct {
	char name[20];
	int numdevs;
	struct DEV *dev;
} mfr[] = {
	{ "MICROCHIP", sizeof(PIC_devs)/sizeof(struct DEV), PIC_devs }
};

int mfrno = 0, devno = 0;

#define BUFSIZE		0x8000U

long addr;
long DevStart= 0x0000;
long Counter = 0x0000;
int last_volt;

/*                           123456789012345678901234567 */
const char *memareas[] = {	"                           ",
							"(Program memory)           ",
							"(Data memory)              ",
							"(Program & Data memory)    " };
int area = 3;

void dly2u( void )
{
	dly1u();
	dly1u();
}

void dly5u( void )
{
	dly2u();
	dly2u();
	dly1u();
}

void ShowCounter( long val )
{
	struct text_info ti;

	gettextinfo( &ti );

	textattr( ( BLUE << 4 ) + WHITE );
	locate( 5, 73 ); cprintf( "%04lX", val );

	textattr( ti.attribute );
	gotoxy( ti.curx, ti.cury );
}

void ShowConfig( void )
{
	struct text_info ti;
	int i;

	gettextinfo( &ti );

	textattr( ( BLUE << 4 ) + WHITE );
	locate( 8,54 );
	for( i = 0x4000; i < 0x4008; i += 2 )
		cprintf( "%02X ", buffer[i] );

	locate( 9,63 );
	cprintf( "%04X  ID : %04X",
		*(unsigned*)(buffer+0x400E), i = *(unsigned*)(buffer+0x400C) );

	switch( i & 0x3FE0 )	// mask off revision number (last 5bits)
	{
	case 0x0D00:	if( devno != 7 ) i = 1; break;	// PIC16F870
	case 0x0D20:	if( devno != 8 ) i = 1; break;	// PIC16F871
	case 0x08E0:	if( devno != 9 ) i = 1; break;	// PIC16F872
	case 0x0960:	if( devno != 10) i = 1; break;	// PIC16F873
	case 0x0E40:	if( devno != 11) i = 1; break;	// PIC16F873A
	case 0x0920:	if( devno != 12) i = 1; break;	// PIC16F874
	case 0x0E60:	if( devno != 13) i = 1; break;	// PIC16F874A
	case 0x09E0:	if( devno != 14) i = 1; break;	// PIC16F876
	case 0x0E00:	if( devno != 15) i = 1; break;	// PIC16F876A
	case 0x09A0:	if( devno != 16) i = 1; break;	// PIC16F877
	case 0x0E20:	if( devno != 17) i = 1; break;	// PIC16F877A

	case 0x07A0:	if( devno != 21) i = 1; break;	// PIC16F627
	case 0x1040:	if( devno != 22) i = 1; break;	// PIC16F627A
	case 0x07C0:	if( devno != 23) i = 1; break;	// PIC16F628
	case 0x1060:	if( devno != 24) i = 1; break;	// PIC16F628A

	case 0x1100:	if( devno != 25) i = 1; break;	// PIC16F648A

	case 0x3FE0:	break;							// ID not read yet
	default:		i = 0; break;					// unknown ID
	}

	if( i == 1 )
	{
		locate( 0, 77 ); cprintf( "??" );
	}

	locate( 2, 69 ); cprintf( "%04X", mfr[mfrno].dev[devno].DataSize );

	textattr( ( CYAN<< 4 ) + WHITE );
	locate( 8, 75 ); cprintf( "OFF" );

	textattr( ti.attribute );
	gotoxy( ti.curx, ti.cury );
}

void ShowType( void )
{
	if( mfrno < 0 || mfrno > sizeof(mfr) / sizeof(mfr[0]) )
		mfrno = 0;
	if( devno < 0 || devno >= mfr[mfrno].numdevs )
		devno = 0;

	bufsize = mfr[mfrno].dev[devno].Size;
	BufEnd = bufsize - 1;

	textattr( ( BLUE << 4 ) | WHITE );

	locate( 0,40 ); cprintf( "*Mfr.: %s", mfr[mfrno].name );
	locate( 0,60 ); cprintf( "*TYPE: %s", mfr[mfrno].dev[devno].name );

	locate( 3,41 ); cprintf( "       end   addr.: %04lX", BufEnd );

	textattr( ( CYAN << 4 ) | WHITE );
}

// Check, if Pin can be pulled LOW constantly
static int pinused( int pin )
{
	return( _MCLR == pin || _DATA == pin || _CLK == pin ||
			_VCC1 == pin || _VCC2 == pin );
}

void DeviceSetup( void )
{
	_MCLR = mfr[mfrno].dev[devno].mclr;
	_DATA = mfr[mfrno].dev[devno].dat;
	_CLK  = mfr[mfrno].dev[devno].clk;
	_LVP  = mfr[mfrno].dev[devno].lvp;

	_UB   = mfr[mfrno].dev[devno].UB;
	_VPP  = mfr[mfrno].dev[devno].VPP;
	
	_VCC1 = mfr[mfrno].dev[devno].vcc1;
	_VCC2 = mfr[mfrno].dev[devno].vcc2;
	
	_GND1 = mfr[mfrno].dev[devno].gnd1;
	_GND2 = mfr[mfrno].dev[devno].gnd2;

	// Can't use Pins 2,3,4,6 and 8 for neither Vop nor Vhh
	if( _MCLR == 2 || _MCLR == 3 || _MCLR == 4 || _MCLR == 6 || _MCLR == 8 )
	{
		errbeep();
		textattr( ( RED << 4 ) | WHITE );
		locate( 23, 41 ); cprintf( "Connect socket pin 1 to IC pin %d", _MCLR );
		textattr( ( CYAN << 4 ) | WHITE );
		_MCLR = 1;									// force using Pin 1
	}

	// Check, if Pin20 is a pin we can pull LOW all the time
	if( !pinused( 20 ) )
		setport( OTHERENID, 0, 0 );			// Pin 20 = GND
	
	// If not: check, if Pins 11 and 30 can be pulled LOW constantly
	else if( !pinused( 11 ) && !pinused( 30 ) )
		setport( OTHERENID, 0, 1 );			// Pin 11,30 = GND

	// Else emit warning, force adapter usage with Pin20 low
	else
	{
		errbeep();
		textattr( ( RED << 4 ) | WHITE );
		locate( 41, 23 ); cprintf( "Use adapter ADP-MPU4" );
		textattr( ( CYAN << 4 ) | WHITE );
		setport( OTHERENID, 0, 0 );				// Pin 20 = GND
	}

	if( mfr[mfrno].dev[devno].DataSize == 0 )
		area = 0;
	else
		area = 3;
}

void power( int voltage, int vpp )
{
	int i;

	// we dont need VHH
	setdac( VHHID, 0 );										// VHH = 0V
	for( i = 0; i <= 4; ++i ) setport( VHHENID, i, 0 );		// no VHH
	setport( VHHENCID,0, 0 );								// no VHHC
	setport( VHHENCID,1, 0 );								// no VHHC

	if( voltage )
	{
		// Set all pins LOW
		for( i = 1; i <= 40; ++i )
		{
			setpin( i, VOPENID, 0 );			// remove Vpp
			setpin( i, TTLID, 0 );				// set low
		}

		if( !vpp && !_LVP )
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			locate( 23, 41 ); cprintf( "ERROR: neither Vpp nor LVP mode" );
			textattr( ( CYAN << 4 ) | WHITE );
		}

		if( !_LVP && _VPP )						// chip has no LVP mode
			vpp = _VPP;							// or we don't want it...

		if( mfr[mfrno].dev[devno].VPP1st &&		// apply VPP before VCC
			vpp )
		{
			setdac( VOPID, vpp );				// raise Vpp to xV
			setdac( VCCID, voltage );			// raise Vcc to xV
			last_volt = voltage;
			delay(100);							// let stabilize

			setpin( _MCLR, TTLID, 1 );			// remove reset from MCLR
			setpin( _MCLR, VOPENID, 1 );		// apply Vpp to it
			dly10u();

			setpin( _VCC1, TTLID, 1 );			// apply Vcc1 & Vcc2
			setpin( _VCC2, TTLID, 1 );
			setpin( _VCC1, VCCENID, 1 );
			setpin( _VCC2, VCCENID, 1 );
			
			dly10m();
		}
		else									// apply VCC, then VPP
		{
			setdac( VCCID, voltage );			// raise Vcc to xV
			setdac( VOPID, vpp );				// raise Vpp to xV
			last_volt = voltage;
			delay( 100 );						// let stabilize

			setpin( _VCC1, TTLID, 1 );			// apply Vcc1 & Vcc2
			setpin( _VCC2, TTLID, 1 );
			setpin( _VCC1, VCCENID, 1 );
			setpin( _VCC2, VCCENID, 1 );
			dly10u();

			if( vpp )
			{
				setpin( _MCLR, TTLID, 1 );		// remove reset from MCLR
				setpin( _MCLR, VOPENID, 1 );	// .. and apply Vpp to it
				dly10u();
			}
			else
			{
				setpin( _MCLR, TTLID, 1 );		// remove reset from MCLR
				setpin( _LVP, TTLID, 1 );		// activate LVP mode
			}
		}

		locate( 2, 75 );
		textattr( ( BLUE << 4 ) | WHITE );
		cprintf( "%s", (!vpp && _LVP) ? "LVP" : ( vpp ? "HVP" : "???" ) );
		textattr( ( CYAN << 4 ) | WHITE );

		delay( 100 );							// wait to stabilize
	}
	else
	{
		setpin( _LVP,  TTLID, 0 );				// remove LVP voltage

		setpin( _MCLR, VOPENID, 0 );			// remove Vpp
		setpin( _MCLR, TTLID, 0 );				// and reset chip

		setdac( VOPID, 0 );						// Vpp = 0V
		setdac( VCCID, 0 );						// Vcc = 0V

		setpin( _VCC1, VCCENID, 0 );			// disable Vcc(s)
		setpin( _VCC2, VCCENID, 0 );

		// Set all pins LOW
		for( i = 1; i <= 40; ++i )
			setpin( i, TTLID, 0 );
	}
}

void pic_command( int c )
{
	int i;

	for( i = 0; i < 6; ++i )		// 6 bits, LSB first
	{
		setpin( _CLK, TTLID, 1 ); dly10u();
		setpin( _DATA, TTLID, c & 1 ); dly10u();
		c >>= 1;
		setpin( _CLK, TTLID, 0 ); dly10u();
	}
	dly50u();
}

void pic_data( unsigned d )
{
	int i;

	d = ( d & 0x3FFF ) << 1;

	for( i = 0; i < 16; ++i )		// START=L, 14 bits (LSB first), STOP=L
	{
		setpin( _CLK, TTLID, 1 ); dly10u();
		setpin( _DATA, TTLID, d & 1 ); dly10u();
		d >>= 1;
		setpin( _CLK, TTLID, 0 ); dly10u();
	}
	dly50u();
}

unsigned pic_read( int seg /* 0: CODE, 1: DATA(EEPROM) */ )
{
	unsigned i, d = 0;

	pic_command( seg ? PIC_READ_DATA : PIC_READ_CODE );

	setpin( _DATA, TTLID, 1 ); dly10u(); // make pin an input and wait a bit

	for( i = 0; i < 16; ++i )
	{
		setpin( _CLK, TTLID, 1 ); dly10u();
		d |= ( getpin( _DATA ) ? 0x8000 : 0 );		// LSB first
		dly10u();
		d >>= 1;
		setpin( _CLK, TTLID, 0 ); dly10u();
	}
	dly50u();
	return d & 0x3FFF;				// Bit0 and 15 where read from HiZ...
}

void reset( int vpp )
{
	power( 0, 0 );
	delay( 100 );
	power( last_volt, vpp );
}

void do_increment( int num )
{
	while( num-- )
	{
		pic_command( PIC_INCREMENT );
		if( addr < 0x4200 ) addr += 2;
		else ++addr;
		dly50u();
	}
}

void do_load( int seg, unsigned val )
{
	pic_command( seg ); pic_data( val );
}

int do_prog1( unsigned lastval )
{
	int num, end = 25, step = 1;
	unsigned mask;

	mask = mfr[mfrno].dev[devno].cfg_mask;

	switch( mfr[mfrno].dev[devno].prog_type )
	{
	case 0:		// max. 25 pulses of 100us + 3*n additional pulses

		for( num = 1; num != end; num += step )
		{
			pic_command( PIC_START_PROG );
			dly100u();
			pic_command( PIC_END_PROG );

			if( step == 1 )
			{
				if( addr >= 0x4200 )		// DATA
				{
					if( ( pic_read( 1 ) & 0xFF ) == lastval )
					{
						num *= 3; end = 0; step = -1;
					}
				}
				else if( addr == 0x400E )	// CONFIG word
				{
					if( ( ( pic_read( 0 ) ^ lastval ) & mask ) == 0 )
					{
						num *= 3; end = 0; step = -1;
					}
				}
				else if( pic_read( 0 ) == lastval )	// CODE & ID
				{
					num *= 3; end = 0; step = -1;
				}
			}
		}
		return num;

	case 1:		// PROG_ONLY, wait 4ms (10ms), no END (program only)
		pic_command( PIC_PROG_ONLY );
		delay(20);
		if( addr >= 0x4200 )
			return( ( pic_read( 1 ) & 0xFF ) == lastval );
		else if( addr == 0x400E )
			return( ( pic_read( 0 ) ^ lastval ) & mask ) == 0;
		else
			return( pic_read( 0 ) == lastval );

	case 2:		// START_PROG, wait 8ms (10ms), no END (erase/program)
		pic_command( PIC_START_PROG );
		delay(20);
		if( addr >= 0x4200 )
			return( ( pic_read( 1 ) & 0xFF ) == lastval );
		else if( addr == 0x400E )
			return( ( pic_read( 0 ) ^ lastval ) & mask ) == 0;
		else
			return( pic_read( 0 ) == lastval );

	case 3:		// PROG_ONLY, wait 20ms, END_PROG
		pic_command( PIC_PROG_ONLY );
		dly20m();
		pic_command( PIC_END_PROG );
		if( addr >= 0x4200 )
			return( ( pic_read( 1 ) & 0xFF ) == lastval );
		else if( addr == 0x400E )
			return( ( pic_read( 0 ) ^ lastval ) & mask ) == 0;
		else
			return( pic_read( 0 ) == lastval );
	}

	return 0;
}

int erase( void )
{
//	reset();

	switch( mfr[mfrno].dev[devno].erase_type )
	{
	case 0:	// Ooops, no ERASE function

		return 0;

	case 1:

		switch( area )
		{
		case 0:	/* none */
		case 1:	/* CODE only */
			do_load( CODE_SEG, 0x3FFF );
			break;

		case 2:	/* DATA only */
			do_load( DATA_SEG, 0x3FFF );
			break;

		case 3:	/* CHIP */
			do_load( CFG_SEG, 0x3FFF );
			do_increment( 7 );			// advance to 0x2007 = CFG WORD
			break;
		}

		pic_command( PIC_BULK_SETUP1 );
		pic_command( PIC_BULK_SETUP2 );
		pic_command( PIC_START_PROG ); dly10m();
		pic_command( PIC_BULK_SETUP1 );
		pic_command( PIC_BULK_SETUP2 );
		break;

	case 2:

		switch( area )
		{
		case 0:	/* none */
		case 1:	/* CODE only */
			pic_command( PIC_CODE_ERASE );
			break;

		case 2:	/* DATA only */
			pic_command( PIC_DATA_ERASE );
			break;

		case 3:	/* CHIP */
			pic_command( PIC_CHIP_ERASE );
			break;
		}
		pic_command( PIC_START_PROG );
		dly10m();
		break;

	case 3:

		switch( area )
		{
		case 0:	/* none */
		case 1:	/* CODE only */

			do_load( CODE_SEG, 0x3FFF );
			pic_command( PIC_CODE_ERASE );
			pic_command( PIC_START_PROG );
			dly10m();
			break;

		case 2:	/* DATA only */

			do_load( DATA_SEG, 0x3FFF );
			pic_command( PIC_DATA_ERASE );
			pic_command( PIC_START_PROG );
			dly10m();
			break;

		case 3:	/* CHIP */

			do_load( CFG_SEG, 0x3FFF );
			do_increment( 7 );			// advance to 0x2007 = CFG WORD
			pic_command( PIC_BULK_SETUP1 );
			pic_command( PIC_BULK_SETUP2 );
			pic_command( PIC_START_PROG ); dly20m();
			pic_command( PIC_BULK_SETUP1 );
			pic_command( PIC_BULK_SETUP2 );

			reset( _VPP );

			do_load( CODE_SEG, 0x3FFF );
			pic_command( PIC_CODE_ERASE );
			pic_command( PIC_START_PROG );
			dly10m();
			
			do_load( DATA_SEG, 0x3FFF );
			pic_command( PIC_DATA_ERASE );
			pic_command( PIC_START_PROG );
			dly10m();

			break;
		}
		break;
	}
	return 1;
}

int check( void )		// BLANK check, return 1 on success
{
	unsigned end, val;
	
	if( ( end = mfr[mfrno].dev[devno].DataSize ) != 0 )
		end += 0x4200;
	else
		end = 0x4010;

	for( addr = BufStart; addr < end; )
	{
		if( ( addr & 0xFF ) == 0 ) ShowCounter( addr );

		if( addr < 0x4200 )					// CODE & CONFIG
		{
			val = pic_read(0) ^ 0x3FFF;

			if( addr == 0x400E )
				val &= mfr[mfrno].dev[devno].cfg_mask;

			if( val )
			{
				ShowCounter( addr );
				return 0;
			}
		}
		else if( ( pic_read(1) & 0xFF ) != 0xFF )	// DATA
		{
			ShowCounter( addr );
			return 0;
		}

		do_increment( 1 );

		if( addr == bufsize )				// continue in CONFIG space
		{
			do_load( CFG_SEG, 0x3FFF );		// goto 0x2000 (0x4000)
			addr = 0x4000;
		}

		else if( addr == 0x4008 )			// ID locations checked
			do_increment( 3 );				// advance to CFG WORD location

		else if( addr == 0x4010 )			// config space checked,
		{
			reset( _VPP );					// reset device and check DATA
			addr = 0x4200;
		}
		
	}
	ShowCounter( addr );
	return 1;
}

int read_verify( int verify )
{
	unsigned val, end, res = 1;
	
	if( ( end = mfr[mfrno].dev[devno].DataSize ) != 0 )
		end += 0x4200;
	else
		end = 0x4010;

	for( addr = BufStart; addr < end; )
	{
		if( ( addr & 0xFF ) == 0 ) ShowCounter( addr );

		if( verify )
		{
			if( addr < 0x4200 )
			{
				val = buffer[addr];
				val |= ( buffer[addr+1] & 0x3F ) << 8;
				val ^= pic_read(0);

				if( addr == 0x400E )
					val &= mfr[mfrno].dev[devno].cfg_mask;
					
				if( val )
				{
					ShowCounter( addr );
					res = 0;
					break;
				}
			}
			else if( ( pic_read(1) & 0xFF ) != buffer[addr] )
			{
				ShowCounter( addr );
				res = 0;
				break;
			}
		}
		else
		{
			if( addr < 0x4200 )
			{
				val = pic_read(0);
				Chks += ( buffer[addr] = val & 0xFF );
				Chks += ( buffer[addr+1] = ( val >> 8 ) & 0x3F );
			}
			else
				Chks += ( buffer[addr] = pic_read(1) & 0xFF );
		}

		do_increment( 1 );

		if( addr == bufsize )				// continue in CONFIG space
		{
			do_load( CFG_SEG, 0x3FFF );		// goto 0x2000 (0x4000)
			addr = 0x4000;
		}

		else if( addr == 0x4008 )			// ID locations done
			do_increment( verify ? 3 : 2 );	// advance to DEVID or CFG word

		else if( addr == 0x4010 )			// config space done,
		{
			reset( _VPP );					// reset device and check DATA
			addr = 0x4200;
		}
	}
	ShowCounter( addr );

	return res;
}

int program( void )
{
	int i, n, val;
	unsigned end;
	
	if( ( end = mfr[mfrno].dev[devno].DataSize ) != 0 )
		end += 0x4200;
	else
		end = 0x4010;

	n = mfr[mfrno].dev[devno].NumProgBytes;

	for( addr = BufStart; addr < end; )
	{
		if( ( addr & 0xFF ) == 0 ) ShowCounter( addr );

		i = n;

		while( i-- )
		{
			if( addr < 0x4200 )				// CODE & CONFIG space
			{
				val = buffer[addr];
				val |= ( buffer[addr+1] & 0x3F ) << 8;
				do_load( CODE_SEG, val );
			}
			else
			{
				val = buffer[addr];
				do_load( DATA_SEG, val );
			}

			if( i ) do_increment( 1 );		// don't increment after n-th load
		}

		if( !do_prog1( val ) )
			return 0;

		do_increment( 1 );

		if( addr == bufsize )				// CODE space done
		{
			do_load( CFG_SEG, 0x3FFF );		// goto 0x2000 (0x4000)
			addr = 0x4000;
		}

		else if( addr == 0x4008 )			// ID locations checked
			do_increment( 3 );				// advance to CFG WORD location

		else if( addr == 0x4010 )			// CONFIG space done,
		{
			reset( _VPP );					// reset device and do DATA
			addr = 0x4200;
		}
	}
	ShowCounter( addr );
	return 1;
}

int config( void )
{
	int i, n, val;
	
	if( ( n = mfr[mfrno].dev[devno].NumProgBytes ) > 4 )
		n = 4;

	do_load( CFG_SEG, 0x3FFF );		// goto 0x2000 (0x4000)

	for( addr = 0x4000; addr < 0x4010; )
	{
		i = n;
		while( i-- )
		{
			val = buffer[addr];
			val |= ( buffer[addr+1] & 0x3F ) << 8;
			do_load( CODE_SEG, val );
			if( i ) do_increment( 1 );	// don't increment after n-th load
		}

		if( !do_prog1( val ) )
			return 0;

		do_increment( 1 );

		if( addr == 0x4008 )
			do_increment( 3 );		// advance to CFG word
	}
	return 1;
}

int flash_check( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   BLANK CHECK device:" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to check (Y/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );
		
		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Blank checking now... " );

		power( _UB, _VPP ); done = check(); power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Blank checking now... " );

		locate( 15, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Blank check error at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
		}
	}
}

void flash_program( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   PROGRAM :" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to program (Y/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Programming now... " );

		power( _UB, _VPP ); done = program(); power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Programming now... " );

		locate( 15, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			setport( USERBITS, 0, 8 );
			cprintf( " OK !" );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Program error ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
		}
	}
}

void flash_config( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 42 ); cprintf( " Program ID & CFG & protect bits: " );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to program (Y/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Programming now... " );

		power( _UB, _VPP ); done = config(); power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Programming now... " );

		locate( 15, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			setport( USERBITS, 0, 8 );
			cprintf( " OK !" );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Program error ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
		}
	}
}

void flash_read( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   READ to buffer :" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to start (Y/Even/Odd/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;

			case 'e':
			case 'E':
			case 'o':
			case 'O':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Reading now... " );

		Chks = 0;
		power( _UB, _VPP ); read_verify(0); power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Reading now... " );

		locate( 15, 41 );
		putchar( 7 );
		cprintf( " OK !" );

		textattr( ( BLUE << 4 ) | WHITE );
		locate( 4, 41 ); cprintf( "       Check Sum  : %04X", Chks );

		ShowConfig();
	}
}

void flash_verify( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   VERIFY with buffer :" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to verify (Y/Even/Odd/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;

			case 'e':
			case 'E':
			case 'o':
			case 'O':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Verifying now @ VDDmin... " );

		power( mfr[mfrno].dev[devno].UBmin, _VPP );
		done = read_verify(1);
		power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Verifying now @ VDDmin... " );

		locate( 15, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " VERIFY ERROR ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
			continue;
		}

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 16, 41 ); cprintf( "Verifying now @ VDDmax... " );

		power( mfr[mfrno].dev[devno].UBmax, _VPP );
		done = read_verify(1);
		power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 16, 41 ); cprintf( "Verifying now @ VDDmax... " );

		locate( 17, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " VERIFY ERROR ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
		}
	}
}

void flash_erase( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "  EEPROM Erase:" );

	if( !mfr[mfrno].dev[devno].erase_type )
	{
		errbeep();
		textattr( ( RED << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "No ERASE function" );
		textattr( ( CYAN << 4 ) | WHITE );

		locate( 21, 41 ); cprintf( "press any key to continue" );
		if( getch() == 0 ) getch();
		return;
	}

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to erase (Y/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Erase now... " );

		power( _UB, _VPP ); done = erase(); power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Erase now... " );

		locate( 15, 41 );
		if( done )
		{
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " ERROR ! " );
			textattr( ( CYAN << 4 ) | WHITE );

			locate( 21, 41 ); cprintf( "press any key to continue" );
			if( getch() == 0 ) getch();
		}

		clscrn( 13,41, 22,78 );
	}
}

void flash_auto( void )
{
	int done, l;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   AUTO :" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to start (Y/Even/Odd/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;

			case 'e':
			case 'E':
			case 'o':
			case 'O':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		l = 14;

		// BLANK CHECK

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( l, 41 ); cprintf( "Blank checking now... " );

		power( _UB, _VPP ); done = check(); power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( l++, 41 ); cprintf( "Blank checking now... " );

		locate( l++, 41 );

		if( done )
		{
			ShowCounter( BufEnd );
			cprintf( " OK !" );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Blank check error at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );

		// ERASE

			l -= 2;

			clscrn( l, 41, l+1, 78 );

			if( !mfr[mfrno].dev[devno].erase_type )
			{
				errbeep();
				textattr( ( RED << 4 ) | WHITE );
				cprintf( "No ERASE function" );
				textattr( ( CYAN << 4 ) | WHITE );

				continue;
			}

			textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
			locate( l, 41 ); cprintf( "Erase now... " );

			power( _UB, _VPP ); done = erase(); power( 0, 0 );

			textattr( ( CYAN << 4 ) | WHITE );
			locate( l++, 41 ); cprintf( "Erase now... " );

			locate( l++, 41 );

			if( done )
				cprintf( " OK !" );
			else
			{
				errbeep();
				textattr( ( RED << 4 ) | WHITE );
				cprintf( " ERROR" );
				textattr( ( CYAN << 4 ) | WHITE );

				continue;
			}
		}

		// PROGRAM

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( l, 41 ); cprintf( "Programming now... " );

		power( _UB, _VPP ); done = program(); power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( l++, 41 ); cprintf( "Programming now... " );

		locate( l++, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			cprintf( " OK !" );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Program error ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );

			continue;
		}

		// VERIFY

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( l, 41 ); cprintf( "VDD max verifying now..." );

		power( mfr[mfrno].dev[devno].UBmax, _VPP );
		done = read_verify(1);
		power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( l++, 41 ); cprintf( "VDD max verifying now..." );

		locate( l++, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " VERIFY ERROR ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
			continue;
		}

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( l, 41 ); cprintf( "VDD min verifying now..." );

		power( mfr[mfrno].dev[devno].UBmin, _VPP );
		done = read_verify(1);
		power( 0, 0 );

		textattr( ( CYAN << 4 ) | WHITE );
		locate( l++, 41 ); cprintf( "VDD min verifying now..." );

		locate( l, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
			putchar( 7 );
		}
		else
		{
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " VERIFY ERROR ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
			errbeep();
		}
	}
}

void change_area( void )
{
	if( mfr[mfrno].dev[devno].DataSize == 0 )
		area = 0;
	else if( area >= 3 )
		area = 1;		/* CODE only */
	else if( area == 1 )
		area = 2;		/* DATA only */
	else if( area == 2 )
		area = 3;		/* CODE+DATA */
}

void edit_config( void )
{
	static char *osctype[4] = { "LP","XT","HS","RC" };
	int i, done;
	unsigned val, cfg = *(unsigned*)( buffer + 0x400E );

	textattr( ( CYAN << 4 ) | WHITE );
	_window(  1,0, 10,39 );
	_window( 11,1, 24,78 );

	locate(  2, 1 ); cprintf( "Current Oscillator Selection :" );
	locate(  3, 1 ); cprintf( "      Watchdog :" );
	locate(  4, 1 ); cprintf( "Power-up timer :" );
	locate(  5, 1 ); cprintf( "CODE Protection:" );
	locate(  6, 1 ); cprintf( "DATA Protection:" );
	locate(  7, 1 ); cprintf( "User FLASH pgm :" );
	locate(  8, 1 ); cprintf( "Brownout reset :" );
	locate(  9, 1 ); cprintf( "Low voltage pgm:" );

	locate( 12, 4 ); cprintf( "A : Oscillator option toggle" );
	locate( 12,43 ); cprintf( "B : Watchdog Option toggle" );
	locate( 13, 4 ); cprintf( "C : Power-up timer Option toggle" );
	locate( 13,43 ); cprintf( "D : Brown-out reset Option toggle" );
	locate( 14, 4 ); cprintf( "E : Low voltage pgm Option toggle" );

	locate( 16, 4 ); cprintf( "K : CODE protection toggle" );
	locate( 16,43 ); cprintf( "L : DATA protection toggle" );
	locate( 17, 4 ); cprintf( "M : FLASH User write Option toggle" );
	locate( 17,43 ); cprintf( "N : DEBUG enable toggle" );
 
	locate( 19, 4 ); cprintf( "1 : edit ID code" );
	locate( 19,43 ); cprintf( "2 : set serial on of off" );
	locate( 20, 4 ); cprintf( "3 : read ID code" );
	locate( 20,43 ); cprintf( "4 : program ID code" );
	locate( 21, 4 ); cprintf( "5 : read configuration code" );
	locate( 21,43 ); cprintf( "6 : program configuration code" );

	locate( 23, 4 ); cprintf( "Select options or <CR><ESC> to go back to the main menu ?" );

	textattr( ( BLUE << 4 ) | WHITE );
	locate(  1, 5 ); cprintf( " Configuration Bit Setting :" );
	locate( 11,28 ); cprintf( " Configuration Options :" );

	for(;;)
	{
		textattr( ( BLUE << 4 ) | WHITE );

		locate( 9,63 ); cprintf( "%04X", cfg );

		locate( 2,32 ); /* OSC */ cprintf( osctype[ cfg & 0x003 ] );
		locate( 3,18 ); /* WDT */ cprintf( (cfg & 0x004) ? "Enabled " : "Disabled" );
		locate( 4,18 ); /* PWT */ cprintf( (cfg & 0x008) ? "Disabled" : "Enabled " );
		locate( 5,18 ); /* CPT */
		switch( ( cfg & 0x030 ) >> 4 )
		{
		case 0:	cprintf( "All protected" ); break;
		case 1:	cprintf( " %04lX - %04lX ", bufsize/2, bufsize-1 ); break;
		case 2: cprintf( " %04X - %04lX ", 0, bufsize/2 - 1 ); break;
		case 3: cprintf( "Not protected" ); break;
		}
		locate( 6,18 ); cprintf( (cfg & 0x100) ? "Disabled" : "Enabled " );
		locate( 7,18 ); cprintf( (cfg & 0x200) ? "Enabled " : "Disabled" );
		locate( 8,18 ); cprintf( (cfg & 0x040) ? "Enabled " : "Disabled" );
		locate( 9,18 ); cprintf( (cfg & 0x080) ? "Enabled " : "Disabled" );

		for( done = 0; !done; )
		{
			done = 1;

			switch( toupper( getch() ) )
			{
			case 0:	getch(); done = 0; break;

			case '\n':
			case '\r':
			case 0x1B:
				*(unsigned*)( buffer + 0x400E ) = cfg;
				return;

			case 'A' : // Oscillator option toggle
				cfg = ( cfg & ~0x003 ) | ( (cfg + 1) & 0x003 ); break;

			case 'B' : // Watchdog Option toggle
				cfg ^= 0x004; break;

			case 'C' : // Power-up timer Option toggle
				cfg ^= 0x008; break;

			case 'D' : // Brown-out reset Option toggle
				cfg ^= 0x040; break;

			case 'E' : // Low voltage pgm Option toggle
				cfg ^= 0x080; break;

			case 'K' : // CODE protection toggle
				cfg = ( cfg & ~0x3030 ) | ( ( (cfg&0x3030) + 0x1010) & 0x3030 ); break;

			case 'L' : // DATA protection toggle
				cfg ^= 0x100; break;

			case 'M' : // FLASH User write Option toggle
				cfg ^= 0x200; break;

			case 'N' : // DEBUG Option toggle
				cfg ^= 0x400; break;

			case '1' : // edit ID code
			case '2' : // set serial on of off

				done = 0; break;

			case '3' : // read ID code
				power( _UB, _VPP );
				do_load( CFG_SEG, 0x3FFF );		// goto 0x2000 (0x4000)
				for( i = 0x4000; i < 0x4008; i += 2 )
				{
					val = pic_read( 0 );
					buffer[ i ] = val & 0xFF;
					buffer[i+1] = ( val >> 8 ) & 0xFF;
					do_increment( 1 );
				}
				power( 0, 0 ); break;

			case '4' : // program ID code
				power( _UB, _VPP );
				do_load( CFG_SEG, 0x3FFF );		// goto 0x2000 (0x4000)
				for( i = 0x4000; i < 0x4008; i += 2 )
				{
					int n = mfr[mfrno].dev[devno].NumProgBytes;
					while( n-- )
					{
						val = buffer[ i ];
						val |= ( buffer[i+1] << 8 );
						do_load( CODE_SEG, val );
						if( n ) do_increment( 1 );
					}
					do_prog1( val );
					do_increment( 1 );
				}
				power( 0, 0 ); break;

			case '5' : // read configuration code
				power( _UB, _VPP );
				do_load( CFG_SEG, 0x3FFF );		// goto 0x2000 (0x4000)
				do_increment( 7 );
				cfg = pic_read( 0 );
				power( 0, 0 ); break;

			case '6' : // program configuration code
				power( _UB, _VPP );
				do_load( CFG_SEG, cfg );		// goto 0x2000 (0x4000)
				do_increment( 7 );
				do_prog1( cfg );
				power( 0, 0 ); break;

			default: done = 0;
			}
		}
	}	
}


void type_select( void )
{
	int done, i, num, len=15, left = 40;
	char no[10];

	if( ( num = mfr[mfrno].numdevs ) > 14 )
	{
		left = 0;
		for( i = len = 0; i < num; ++i )
			if( ( done = strlen( mfr[mfrno].dev[i].name ) ) > len )
				len = done;
	}

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,left, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, left+5 ); cprintf( "  TYPE SELECT:" );

	textattr( ( CYAN << 4 ) | WHITE );

	for( i = 0; i < num; ++i )
	{
		locate( 13+(i%7), left + 1 + (i/7)*(len+4) );
		cprintf( "%d.%s", i, mfr[mfrno].dev[i].name );
	}

	locate( 21, left+1 ); cprintf( "<CR> back to main menu." );
	locate( 22, left+1 ); cprintf( "SELECT NUMBER ?" );

	for(;;)
	{
		for( no[0] = done = 0; !done; )
		{
			locate( 22, left+16 ); cprintf( "%s ", no );
			locate( 22, left+16+strlen(no) );

			switch( i = getch() )
			{
			case 0:
				getch();
				break;

			case 8:
				if( ( i = strlen(no) ) != 0 )
					no[i-1] = 0;
				break;

			case '\n':
			case '\r':
				if( strlen( no ) &&
					( i = atoi(no) ) >= 0 && i < mfr[mfrno].numdevs )
				{
					devno = i;
					ShowType();
					return;
				}
				break;

			case 0x1B:
				return;

			default:
				if( isdigit( i ) )
					strcat( no, (char*)&i );
				break;
			}
		}
	}
}

void mfr_select( void )
{
	int done, i;
	char no[10];

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "  MFR SELECT:" );

	textattr( ( CYAN << 4 ) | WHITE );

	for( i = 0; i < sizeof(mfr) / sizeof(mfr[0]); ++i )
	{
		locate( 13+i, 41 ); cprintf( "%d.%s", i, mfr[i].name );
	}

	locate( 21, 41 ); cprintf( "<CR> back to main menu." );
	locate( 22, 41 ); cprintf( "SELECT NUMBER ?" );

	for(;;)
	{
		for( no[0] = done = 0; !done; )
		{
			locate( 22, 56 ); cprintf( "%s ", no );
			locate( 22, 56+strlen(no) );

			switch( i = getch() )
			{
			case 0:
				getch();
				break;

			case 8:
				if( ( i = strlen(no) ) != 0 )
					no[i-1] = 0;
				break;

			case '\n':
			case '\r':
				if( strlen( no ) &&
					( i = atoi(no) ) >= 0 && i < sizeof(mfr) / sizeof(mfr[0]) )
				{
					mfrno = i;
					if( devno >= mfr[mfrno].numdevs )
						devno = 0;
					ShowType();
					return;
				}
				break;

			case 0x1B:
				return;

			default:
				if( isdigit( i ) )
					strcat( no, (char*)&i );
				break;
			}
		}
	}
}

/*=========================================================*/
int main( void )
{
	int ch = 0, redraw, first = 1;
	long tmpval;

	/*---------------------------------------------------------*/
	/* main program starts here                                */
	/*---------------------------------------------------------*/

	getcwd( oldpath, 260 );
	strcpy( path, oldpath );

	if( ( buffer = farmalloc( BUFSIZE ) ) == NULL )
		return -1;

	memset( (void far *)buffer, 0, BUFSIZE );

	ReadConfig();
	delay(0);

	for( ;; )
	{
		if( first )
		{
			first = 0;

			area = 3;

			init_hw();
			initdacs();
			setport( USERBITS, 0, 0 );

			DeviceSetup();
		}

		textattr( ( LIGHTGRAY << 4 ) | YELLOW ); clscrn( 0,0, 24,79 );

		locate( 0,0 ); cprintf( "Universal   Programmer" );
		locate( 1,0 ); cprintf( "MODEL: PC Based" );
		locate( 2,0 ); cprintf( "MPU PIC16 section  " _VERSION_ );

		textattr( ( BLUE << 4 ) + WHITE ); clscrn( 0,40, 6,79 );

		ShowType();

		textattr( ( BLUE << 4 ) + WHITE ); _window( 1,40, 6,79 );
		locate( 1,53 ); cprintf( " TARGET ZONE " );
		locate( 2,41 ); cprintf( "Buffer start addr.: %04lX", BufStart );
		locate( 3,41 ); cprintf( "       end   addr.: %04lX", BufEnd );
		locate( 4,41 ); cprintf( "       Check Sum  : %04X", Chks );
		locate( 5,41 ); cprintf( "Device start addr.: %04lX", DevStart );

		_window( 3,69, 6,79 );
		locate( 4,71 ); cprintf( "COUNTER" );
		ShowCounter( 0 );

		_window( 7,40, 10,79 );
		locate( 7,43 ); cprintf( " Device ID & Configuration bits " );
		locate( 8,42 ); cprintf( "ID3 - ID0 :" );
		locate( 8,42 ); cprintf( "Lock Bits :" );
		locate( 8,67 ); cprintf( "*Fuses :" );
		locate( 9,42 ); cprintf( "Configuration bits : " );
		ShowConfig();

		textattr( ( CYAN << 4 ) | WHITE ); clscrn( 3,0, 23,38 );

		locate( 3,0 ); cprintf( "-------------  Main Menu  -------------" );
		locate( 4,0 ); cprintf( "1. DOS SHELL                           " );
		locate( 5,0 ); cprintf( "2. Load BIN or HEX file to buffer      " );
		locate( 6,0 ); cprintf( "3. Save buffer to disk                 " );
		locate( 7,0 ); cprintf( "4. Edit buffer     7. Display buffer   " );
		locate( 8,0 ); cprintf( "5. Change I/O base address             " );
		locate( 9,0 ); cprintf( "6. Display loaded file history         " );
		locate(10,0 ); cprintf( "W. Swap hi-low bytes in buffer         " );
		locate(11,0 ); cprintf( "T. Type select     Z. Target zone      " );
		locate(12,0 ); cprintf( "B. Blank check     D. Display          " );
		locate(13,0 );
		if( mfr[mfrno].dev[devno].erase_type == 0 )
					   cprintf( "                                       " );
		else
					   cprintf( "S. Erase %s   ", memareas[area] );

		locate(14,0 ); cprintf( "P. Program %s ", memareas[area] );
		locate(15,0 );
		if( mfr[mfrno].dev[devno].erase_type == 0 )
					   cprintf( "A. Auto(B&P&V&L)   " );
		else
					   cprintf( "A. Auto(B&S&P&V&L) " );

		locate(15,19);
		if( mfr[mfrno].dev[devno].DataSize != 0 )
					   cprintf(                    "X. Change Mem area  " );
		else
					   cprintf(                    "                    " );

		locate(16,0 ); cprintf( "R. Read            V. Verify           " );
		locate(17,0 ); cprintf( "C. Compare and display error           " );
		locate(18,0 ); cprintf( "E. Configuration & ID code function    " );
		locate(19,0 ); cprintf( "L. Program ID & config. & protect bits " );
		locate(20,0 ); cprintf( "Q. Quit                                " );
		locate(21,0 ); cprintf( "---------------------------------------" );
		locate(22,0 ); cprintf( "Allocation Buffer size : %uK bytes", BUFSIZE/1024 );
		if( ( ch = mfr[mfrno].dev[devno].DataSize ) != 0 ) {
		locate(23,0 ); cprintf( "Data memory buffer at 4200 ~ %04X", 0x4200 + ch - 1 );
		}

		for( redraw = 0; !redraw; )
		{
			textattr( ( BLUE << 4 ) + WHITE ); clscrn( 24,0, 24,38 );

			locate( 24,0 ); cprintf( "Select function ? " );

			for(;;)
			{
				if( ( ch = getch() ) != 0 )
					break;
				getch();				 /* neglect extended code */
			}

			switch( ch = toupper(ch) )
			{
			case '1': dos_shell( "" ); redraw = 1; break;
			case '2': tmpval = bufsize; bufsize = 0x8000;
					  memset( (void far *)buffer, 0, BUFSIZE );
					  load_file();
					  bufsize= tmpval;
					  redraw = 1; ShowConfig(); break;
			case '3': save_file(); break;
			case '4': tmpval = bufsize; bufsize = 0x8000;
					  edit_buffer();
					  bufsize= tmpval;
					  redraw = 1; break;
			case '5': set_io_adr(); first = redraw = 1; break;
			case '7': disp_buffer(); redraw = 1; break;

			case 'M': mfr_select(); redraw = first = 1; break;
			case 'T': type_select(); redraw = first = 1; break;
			case 'E': edit_config(); redraw = 1; break;

			case 'X': change_area(); redraw = 1; break;
			case 'R': flash_read(); break;
			case 'B': flash_check(); break;
			case 'S': flash_erase(); break;
			case 'P': flash_program(); break;
			case 'V': flash_verify(); break;
			case 'L': flash_config(); break;
			case 'A': flash_auto(); break;

//			case '\n':
//			case '\r': redraw = 1; break;		// refresh
			}

			setport( USERBITS, 0, 0 );

			if( ch == 'Q' )
			{
				WriteConfig();
				textattr( LIGHTGRAY ); clrscr();
				chdir( oldpath );
				if( buffer ) farfree( (void far *)buffer );
				return( 0 );
			}

			textattr( ( LIGHTGRAY << 4 ) | YELLOW ); clscrn( 11,40, 23,79 );
		}
	}
}
