#include "all03.h"

#include <bios.h>

#define _VERSION_			"1.00"

#define MODE_WRITE		0x0E		// Bit 0 - 3 : P3.3 - P3.7
#define MODE_READ		0x0C
#define MODE_LOCK1		0x0F
#define MODE_LOCK2		0x03
#define MODE_ERASE		0x01
#define MODE_SIGNAT		0x00

unsigned char _RST, _RDY, _PROG, _XTAL;
unsigned char _VCC, _GND, _MODE[4], _DATA[8];

struct DEV {
	char name[20];
	unsigned Size, DataSize;	// ROM & EEPROM sizes in bytes
	char rst,rdy,prog,xtal;		// Pin numbers for RESET,RDY/BUSY,PROG and XTAL
	char vcc, gnd, mode[4];		// Pin numbers for VCC, GND and MODE0..3
	char data[8];				// Pin numbers for data0..7
	unsigned char UBmin, UBmax;	// min. and max. Vcc value
};

struct DEV ATMEL_devs[] = {
//			TYPE		ROMSIZE	EESIZE	rst	rdy	prg	xtl vcc	gnd	md0 md1 md2 md3 d0 d1 d2 d3 d4 d5 d6 d7 Umin Umax
/* 0*/	{ "AT89C1051",	0x0400,	0x0000, 11,	13,	16,	15, 30,	20, 17,	18, 19, 21, 22,23,24,25,26,27,28,29, 30, 60 },
/* 1*/	{ "AT89C2051",	0x0800,	0x0000, 11,	13,	16,	15, 30,	20, 17,	18, 19, 21, 22,23,24,25,26,27,28,29, 30, 60 },
/* 2*/	{ "AT89C4051",	0x1000,	0x0000, 11,	13,	16,	15, 30,	20, 17,	18, 19, 21, 22,23,24,25,26,27,28,29, 30, 60 }
};

struct {
	char name[20];
	int numdevs;
	struct DEV *dev;
} mfr[] = {
	{ "ATMEL", sizeof(ATMEL_devs)/sizeof(struct DEV), ATMEL_devs }
};

int mfrno = 0, devno = 0;

#define BUFSIZE		0x8000U

long addr;
long DevStart= 0x0000;
long Counter = 0x0000;

void ShowCounter( long val )
{
	struct text_info ti;

	gettextinfo( &ti );

	textattr( ( BLUE << 4 ) + WHITE );
	locate( 5, 73 ); cprintf( "%04lX", val );

	textattr( ti.attribute );
	gotoxy( ti.curx, ti.cury );
}

void ShowType( void )
{
	if( mfrno < 0 || mfrno > sizeof(mfr) / sizeof(mfr[0]) )
		mfrno = 0;
	if( devno < 0 || devno >= mfr[mfrno].numdevs )
		devno = 0;

	bufsize = mfr[mfrno].dev[devno].Size;
	BufEnd = bufsize - 1;

	textattr( ( BLUE << 4 ) | WHITE );

	locate( 0,40 ); cprintf( "*Mfr.: %s", mfr[mfrno].name );
	locate( 0,60 ); cprintf( "*TYPE: %s", mfr[mfrno].dev[devno].name );

	locate( 3,41 ); cprintf( "       end   addr.: %04lX", BufEnd );

	textattr( ( CYAN << 4 ) | WHITE );
}

void setmode( int md )
{
	setpin( _MODE[0], TTLID, (md & 1) ? 1:0 );
	setpin( _MODE[1], TTLID, (md & 2) ? 1:0 );
	setpin( _MODE[2], TTLID, (md & 4) ? 1:0 );
	setpin( _MODE[3], TTLID, (md & 8) ? 1:0 );
}

void power( int voltage )
{
	int i;

	_RST = mfr[mfrno].dev[devno].rst;
	_RDY = mfr[mfrno].dev[devno].rdy;
	_PROG= mfr[mfrno].dev[devno].prog;
	_XTAL= mfr[mfrno].dev[devno].xtal;

	_VCC = mfr[mfrno].dev[devno].vcc;
	_GND = mfr[mfrno].dev[devno].gnd;

	for( i = 0; i < 4; ++i )
		_MODE[i]  = mfr[mfrno].dev[devno].mode[i];

	for( i = 0; i < 8; ++i )
		_DATA[i]  = mfr[mfrno].dev[devno].data[i];

	// we dont need VHH
	setdac( VHHID, 0 );										// VHH = 0V
	for( i = 0; i <= 4; ++i ) setport( VHHENID, i, 0 );		// no VHH
	setport( VHHENCID,0, 0 );								// no VHHC
	setport( VHHENCID,1, 0 );								// no VHHC

	// Can't use Pins 2,3,4,6 and 8 for neither Vop nor Vhh
	if( _RST == 2 || _RST == 3 || _RST == 4 || _RST == 6 || _RST == 8 ||
		( _RST > 32 && _RST != 36 ) )
	{
		errbeep();
		textattr( ( RED << 4 ) | WHITE );
		locate( 41, 23 ); cprintf( "Connect pin 1 to %d", _RST );
		textattr( ( CYAN << 4 ) | WHITE );
		_RST = 1;									// force using Pin 1
	}
	setport( OTHERENID, 0, 0 );			// Pin 20 = GND

	if( voltage )
	{
		// Set all pins HIGH
		for( i = 1; i <= 40; ++i )
			setpin( i, TTLID, 1 );

		// Set RST and XTAL pin LOW
		setpin( _RST, TTLID, 0 );
		setpin( _XTAL, TTLID, 0 );

		// Set MODE
		setmode( MODE_ERASE );					// CHIP ERASE preset

		setdac( VCCID, voltage );				// Vcc = xV
		setpin( _VCC, VCCENID, 1 );				// set Vcc pin
		setdac( VOPID, 120 );					// Vpp = 12V
		delay( 500 );							// wait to stabilize

		setpin( _RST, TTLID, 1 );				// RST = H
	}
	else
	{
		setpin( _RST, VOPENID, 0 );				// remove Vpp
		setpin( _RST, TTLID, 0 );				// pull reset low

		setdac( VOPID, 0 );						// Vpp = 0V
		setdac( VCCID, 0 );						// Vcc = 0V

		setpin( _VCC, VCCENID, 0 );				// disable Vcc(s)

		// Set all pins LOW
		for( i = 1; i <= 40; ++i )
			setpin( i, TTLID, 0 );
	}
}

void reset( void )
{
	setpin( _RST, VOPENID, 0 ); dly1u();	// remove Vpp (if not already done)

	setpin( _RST, TTLID, 0 ); dly1u();		// pulse RST H->L->H
	setpin( _RST, TTLID, 1 ); dly10m();
}

int do_erase( void )
{
	setmode( MODE_ERASE );

	setpin( _RST, VOPENID, 1 ); dly10u();	// apply Vpp
	
	setpin( _PROG, TTLID, 0 );				// pulse PROG low for 10ms
	dly10m();
	setpin( _PROG, TTLID, 1 );

	dly10u(); setpin( _RST, VOPENID, 0 );	// remove Vpp
	return 1;
}

int do_read( void )
{
	int i, val = 0;

	for( i = 0; i < 8; ++i )
		setpin( _DATA[i], TTLID, 1 );	// release pin drivers on data pins

	setmode( MODE_READ );

	for( i = 7; i >= 0; --i )
		val = ( val << 1 ) | getpin( _DATA[i] );

	setmode( MODE_ERASE );
	return val;
}

void do_load( int val )
{
	int i;

	for( i = 0; i < 8; ++i )
	{
		setpin( _DATA[i], TTLID, val & 1 );
		val >>= 1;
	}
}

void do_increment( int num )
{
	do
	{
		setpin( _XTAL, TTLID, 1 ); dly1u();
		setpin( _XTAL, TTLID, 0 );
		++addr;
	} while( --num );
}

int do_prog( void )
{
	long t;

	setpin( _RST, VOPENID, 1 ); dly20u();
	
	setpin( _PROG, TTLID, 0 ); dly1u();
	setpin( _PROG, TTLID, 1 ); dly1u();

	t = biostime( 0, 0 );

	while( biostime( 0, 0 ) - t < 2 )	// max. 55..110ms
	{
		if( getpin( _RDY ) )
			break;
		dly100u();
	}

	dly20u(); setpin( _RST, VOPENID, 0 );

	return getpin( _RDY ) ? 1 : 0;
}

int check( void )		// BLANK check, return 1 on success
{
	unsigned end = mfr[mfrno].dev[devno].Size;

	reset();

	for( addr = BufStart; addr < end; )
	{
		if( ( addr & 0xFF ) == 0 ) ShowCounter( addr );

		if( do_read() != 0xFF )
		{
			ShowCounter( addr );
			return 0;
		}

		do_increment( 1 );
	}
	ShowCounter( addr );
	return 1;
}

int program( void )
{
	unsigned end = mfr[mfrno].dev[devno].Size;

	reset();

	setmode( MODE_WRITE );
	
	for( addr = BufStart; addr < end; )
	{
		if( ( addr & 0xFF ) == 0 ) ShowCounter( addr );

		do_load( buffer[addr] );

		if( !do_prog() )
			return 0;

		do_increment( 1 );
	}
	ShowCounter( addr );
	return 1;
}

int protect( void )
{
	setmode( MODE_LOCK1 );
	if( !do_prog() )
		return 0;

	setmode( MODE_LOCK2 );
	return do_prog();
}


int read_verify( int verify )
{
	unsigned end = mfr[mfrno].dev[devno].Size;
	
	reset();

	for( addr = BufStart; addr < end; )
	{
		if( ( addr & 0xFF ) == 0 ) ShowCounter( addr );

		if( verify )
		{
			if( buffer[addr] != do_read() )
			{
				ShowCounter( addr );
				return 0;
			}
		}
		else
			Chks += ( buffer[addr] = do_read() & 0xFF );

		do_increment( 1 );
	}
	ShowCounter( addr );
	return 1;
}

int flash_check( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   BLANK CHECK device:" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to check (Y/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );
		
		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Blank checking now... " );

		power(50); done = check(); power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Blank checking now... " );

		locate( 15, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Blank check error at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
		}
	}
}

void flash_program( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   PROGRAM :" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to program (Y/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Programming now... " );

		power(50); done = program(); power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Programming now... " );

		locate( 15, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			setport( USERBITS, 0, 8 );
			cprintf( " OK !" );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Program error ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
		}
	}
}

void flash_protect( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 42 ); cprintf( " Program ID & CFG & protect bits: " );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to program (Y/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Programming now... " );

		power(50); done = protect(); power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Programming now... " );

		locate( 15, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			setport( USERBITS, 0, 8 );
			cprintf( " OK !" );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Program error ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
		}
	}
}

void flash_read( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   READ to buffer :" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to start (Y/Even/Odd/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;

			case 'e':
			case 'E':
			case 'o':
			case 'O':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Reading now... " );

		Chks = 0;
		power(50); read_verify(0); power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Reading now... " );

		locate( 15, 41 );
		putchar( 7 );
		cprintf( " OK !" );

		textattr( ( BLUE << 4 ) | WHITE );
		locate( 4, 41 ); cprintf( "       Check Sum  : %04X", Chks );
	}
}

void flash_verify( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   VERIFY with buffer :" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to verify (Y/Even/Odd/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;

			case 'e':
			case 'E':
			case 'o':
			case 'O':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Verifying now @ VDDmin... " );

		power(mfr[mfrno].dev[devno].UBmin);
		done = read_verify(1);
		power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Verifying now @ VDDmin... " );

		locate( 15, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " VERIFY ERROR ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
			continue;
		}

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 16, 41 ); cprintf( "Verifying now @ VDDmax... " );

		power(mfr[mfrno].dev[devno].UBmax);
		done = read_verify(1);
		power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 16, 41 ); cprintf( "Verifying now @ VDDmax... " );

		locate( 17, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " VERIFY ERROR ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
		}
	}
}

void flash_erase( void )
{
	int done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "  EEPROM Erase:" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to erase (Y/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Erase now... " );

		power(50); done = do_erase(); power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( 14, 41 ); cprintf( "Erase now... " );

		locate( 15, 41 );
		if( done )
		{
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " ERROR ! " );
			textattr( ( CYAN << 4 ) | WHITE );

			locate( 21, 41 ); cprintf( "press any key to continue" );
			if( getch() == 0 ) getch();
		}

		clscrn( 13,41, 22,78 );
	}
}

void flash_auto( void )
{
	int done, l;

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "   AUTO :" );

	for(;;)
	{
		textattr( ( CYAN << 4 ) | WHITE );
		locate( 13, 41 ); cprintf( "Ready to start (Y/Even/Odd/<CR>)? " );

		for( done = 0; !done; )
		{
			switch( getch() )
			{
			case 0:
				getch();
				break;

			case '\n':
			case '\r':
			case 0x1B:
				return;

			case 'y':
			case 'Y':
				done = 1;

			case 'e':
			case 'E':
			case 'o':
			case 'O':
				done = 1;
			}
		}

		clscrn( 14,41, 22,78 );

		setport( USERBITS, 0, 0 );

		l = 14;

		// BLANK CHECK

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( l, 41 ); cprintf( "Blank checking now... " );

		power(50); done = check(); power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( l++, 41 ); cprintf( "Blank checking now... " );

		locate( l++, 41 );

		if( done )
		{
			ShowCounter( BufEnd );
			cprintf( " OK !" );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Blank check error at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );

		// ERASE

			l -= 2;

			clscrn( l, 41, l+1, 78 );

			textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
			locate( l, 41 ); cprintf( "Erase now... " );

			power(50); done = do_erase(); power(0);

			textattr( ( CYAN << 4 ) | WHITE );
			locate( l++, 41 ); cprintf( "Erase now... " );

			locate( l++, 41 );

			if( done )
				cprintf( " OK !" );
			else
			{
				errbeep();
				textattr( ( RED << 4 ) | WHITE );
				cprintf( " ERROR" );
				textattr( ( CYAN << 4 ) | WHITE );

				continue;
			}
		}

		// PROGRAM

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( l, 41 ); cprintf( "Programming now... " );

		power(50); done = program(); power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( l++, 41 ); cprintf( "Programming now... " );

		locate( l++, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			cprintf( " OK !" );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( "Program error ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );

			continue;
		}

		// VERIFY

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( l, 41 ); cprintf( "VDD max verifying now..." );

		power(mfr[mfrno].dev[devno].UBmax);
		done = read_verify(1);
		power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( l++, 41 ); cprintf( "VDD max verifying now..." );

		locate( l++, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			putchar( 7 );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
		}
		else
		{
			errbeep();
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " VERIFY ERROR ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
			continue;
		}

		textattr( BLINK | ( LIGHTGREEN << 4 ) | WHITE );
		locate( l, 41 ); cprintf( "VDD min verifying now..." );

		power(mfr[mfrno].dev[devno].UBmin);
		done = read_verify(1);
		power(0);

		textattr( ( CYAN << 4 ) | WHITE );
		locate( l++, 41 ); cprintf( "VDD min verifying now..." );

		locate( l, 41 );
		if( done )
		{
			ShowCounter( BufEnd );
			cprintf( " OK !" );
			setport( USERBITS, 0, 8 );
			putchar( 7 );
		}
		else
		{
			textattr( ( RED << 4 ) | WHITE );
			cprintf( " VERIFY ERROR ! at %04lX", addr );
			textattr( ( CYAN << 4 ) | WHITE );
			errbeep();
		}
	}
}

#if(0)
void edit_config( void )
{
	static char *osctype[4] = { "LP","XT","HS","RC" };
	int i, done;

	textattr( ( CYAN << 4 ) | WHITE );
	_window(  1,0, 10,39 );
	_window( 11,1, 24,78 );

	locate(  2, 1 ); cprintf( "CODE Protection:" );
	locate(  3, 1 ); cprintf( "DATA Protection:" );

	locate( 16, 4 ); cprintf( "K : CODE protection toggle" );
	locate( 16,43 ); cprintf( "L : DATA protection toggle" );
 
	locate( 23, 4 ); cprintf( "Select options or <CR><ESC> to go back to the main menu ?" );

	textattr( ( BLUE << 4 ) | WHITE );
	locate(  1, 5 ); cprintf( " Configuration Bit Setting :" );
	locate( 11,28 ); cprintf( " Configuration Options :" );

	for(;;)
	{
		textattr( ( BLUE << 4 ) | WHITE );

		for( done = 0; !done; )
		{
			done = 1;

			switch( toupper( getch() ) )
			{
			case 0:	getch(); done = 0; break;

			case '\n':
			case '\r':
			case 0x1B: return;

			case 'K' : // CODE protection toggle
				cfg = ( cfg & ~0x3030 ) | ( ( (cfg&0x3030) + 0x1010) & 0x3030 ); break;

			case 'L' : // DATA protection toggle
				cfg ^= 0x100; break;

			default: done = 0;
			}
		}
	}	
}
#endif

void type_select( void )
{
	int done, i, num, len=15, left = 40;
	char no[10];

	if( ( num = mfr[mfrno].numdevs ) > 14 )
	{
		left = 0;
		for( i = len = 0; i < num; ++i )
			if( ( done = strlen( mfr[mfrno].dev[i].name ) ) > len )
				len = done;
	}

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,left, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, left+5 ); cprintf( "  TYPE SELECT:" );

	textattr( ( CYAN << 4 ) | WHITE );

	for( i = 0; i < num; ++i )
	{
		locate( 13+(i%7), left + 1 + (i/7)*(len+4) );
		cprintf( "%d.%s", i, mfr[mfrno].dev[i].name );
	}

	locate( 21, left+1 ); cprintf( "<CR> back to main menu." );
	locate( 22, left+1 ); cprintf( "SELECT NUMBER ?" );

	for(;;)
	{
		for( no[0] = done = 0; !done; )
		{
			locate( 22, left+16 ); cprintf( "%s ", no );
			locate( 22, left+16+strlen(no) );

			switch( i = getch() )
			{
			case 0:
				getch();
				break;

			case 8:
				if( ( i = strlen(no) ) != 0 )
					no[i-1] = 0;
				break;

			case '\n':
			case '\r':
				if( strlen( no ) &&
					( i = atoi(no) ) >= 0 && i < mfr[mfrno].numdevs )
				{
					devno = i;
					ShowType();
					return;
				}
				break;

			case 0x1B:
				return;

			default:
				if( isdigit( i ) )
					strcat( no, (char*)&i );
				break;
			}
		}
	}
}

void mfr_select( void )
{
	int done, i;
	char no[10];

	textattr( ( CYAN << 4 ) | WHITE );
	_window( 12,40, 23,79 );

	textattr( ( BLUE << 4 ) | WHITE );
	locate( 12, 45 ); cprintf( "  MFR SELECT:" );

	textattr( ( CYAN << 4 ) | WHITE );

	for( i = 0; i < sizeof(mfr) / sizeof(mfr[0]); ++i )
	{
		locate( 13+i, 41 ); cprintf( "%d.%s", i, mfr[i].name );
	}

	locate( 21, 41 ); cprintf( "<CR> back to main menu." );
	locate( 22, 41 ); cprintf( "SELECT NUMBER ?" );

	for(;;)
	{
		for( no[0] = done = 0; !done; )
		{
			locate( 22, 56 ); cprintf( "%s ", no );
			locate( 22, 56+strlen(no) );

			switch( i = getch() )
			{
			case 0:
				getch();
				break;

			case 8:
				if( ( i = strlen(no) ) != 0 )
					no[i-1] = 0;
				break;

			case '\n':
			case '\r':
				if( strlen( no ) &&
					( i = atoi(no) ) >= 0 && i < sizeof(mfr) / sizeof(mfr[0]) )
				{
					mfrno = i;
					if( devno >= mfr[mfrno].numdevs )
						devno = 0;
					ShowType();
					return;
				}
				break;

			case 0x1B:
				return;

			default:
				if( isdigit( i ) )
					strcat( no, (char*)&i );
				break;
			}
		}
	}
}

/*=========================================================*/
int main( void )
{
	int ch = 0, redraw, first = 1;
	long tmpval;

	/*---------------------------------------------------------*/
	/* main program starts here                                */
	/*---------------------------------------------------------*/

	getcwd( oldpath, 260 );
	strcpy( path, oldpath );

	if( ( buffer = farmalloc( BUFSIZE ) ) == NULL )
		return -1;

	memset( (void far *)buffer, 0, BUFSIZE );

	ReadConfig();
	delay(0);

	for( ;; )
	{
		if( first )
		{
			first = 0;

			init_hw();
			initdacs();
			setport( USERBITS, 0, 0 );
		}

		textattr( ( LIGHTGRAY << 4 ) | YELLOW ); clscrn( 0,0, 24,79 );

		locate( 0,0 ); cprintf( "Universal   Programmer" );
		locate( 1,0 ); cprintf( "MODEL: PC Based" );
		locate( 2,0 ); cprintf( "MPU 89Cx051 section  " _VERSION_ );

		textattr( ( BLUE << 4 ) + WHITE ); clscrn( 0,40, 6,79 );

		ShowType();

		textattr( ( BLUE << 4 ) + WHITE ); _window( 1,40, 6,79 );
		locate( 1,53 ); cprintf( " TARGET ZONE " );
		locate( 2,41 ); cprintf( "Buffer start addr.: %04lX", BufStart );
		locate( 3,41 ); cprintf( "       end   addr.: %04lX", BufEnd );
		locate( 4,41 ); cprintf( "       Check Sum  : %04X", Chks );
		locate( 5,41 ); cprintf( "Device start addr.: %04lX", DevStart );

		_window( 3,69, 6,79 );
		locate( 4,71 ); cprintf( "COUNTER" );
		ShowCounter( 0 );

		textattr( ( CYAN << 4 ) | WHITE ); clscrn( 3,0, 23,38 );

		locate( 3,0 ); cprintf( "-------------  Main Menu  -------------" );
		locate( 4,0 ); cprintf( "1. DOS SHELL                           " );
		locate( 5,0 ); cprintf( "2. Load BIN or HEX file to buffer      " );
		locate( 6,0 ); cprintf( "3. Save buffer to disk                 " );
		locate( 7,0 ); cprintf( "4. Edit buffer     7. Display buffer   " );
		locate( 8,0 ); cprintf( "5. Change I/O base address             " );
		locate( 9,0 ); cprintf( "6. Display loaded file history         " );
		locate(10,0 ); cprintf( "W. Swap hi-low bytes in buffer         " );
		locate(11,0 ); cprintf( "T. Type select     Z. Target zone      " );
		locate(12,0 ); cprintf( "B. Blank check     D. Display          " );
		if( ( ch = mfr[mfrno].dev[devno].DataSize ) != 0 ) {
		locate(13,0 ); cprintf( "P. Program (Program Mem & Data Mem)    " );
		locate(14,0 ); cprintf( "A. Auto(B&S&P&V&L)                     " );
		locate(15,0 ); cprintf( "S. Erase Program & Data memory         " );
		} else {
		locate(13,0 ); cprintf( "                                       " );
		locate(14,0 ); cprintf( "P. Program         A. Auto(B&S&P&V&L)  " );
		locate(15,0 ); cprintf( "S. Erase Program memory                " );
		}
		locate(16,0 ); cprintf( "R. Read            V. Verify           " );
		locate(17,0 ); cprintf( "C. Compare and display error           " );
		locate(18,0 ); cprintf( "E. Configuration & ID code function    " );
		locate(19,0 ); cprintf( "L. Program ID & config. & protect bits " );
		locate(20,0 ); cprintf( "Q. Quit                                " );
		locate(21,0 ); cprintf( "---------------------------------------" );
		locate(22,0 ); cprintf( "Allocation Buffer size : %uK bytes", BUFSIZE/1024 );
		if( ( ch = mfr[mfrno].dev[devno].DataSize ) != 0 ) {
		locate(23,0 ); cprintf( "Data memory buffer at 4200 ~ %04X", 0x4200 + ch - 1 );
		}

		for( redraw = 0; !redraw; )
		{
			textattr( ( BLUE << 4 ) + WHITE ); clscrn( 24,0, 24,38 );

			locate( 24,0 ); cprintf( "Select function ? " );

			for(;;)
			{
				if( ( ch = getch() ) != 0 )
					break;
				getch();				 /* neglect extended code */
			}

			switch( ch = toupper(ch) )
			{
			case '1': dos_shell( "" ); redraw = 1; break;
			case '2': tmpval = bufsize; bufsize = 0x8000;
					  memset( (void far *)buffer, 0, BUFSIZE );
					  load_file();
					  bufsize= tmpval;
					  redraw = 1; break;
			case '3': save_file(); break;
			case '4': tmpval = bufsize; bufsize = 0x8000;
					  edit_buffer();
					  bufsize= tmpval;
					  redraw = 1; break;
			case '5': set_io_adr(); first = redraw = 1; break;
			case '7': disp_buffer(); redraw = 1; break;

			case 'M': mfr_select(); redraw = first = 1; break;
			case 'T': type_select(); redraw = first = 1; break;
//			case 'E': edit_config(); redraw = 1; break;

			case 'R': flash_read(); break;
			case 'B': flash_check(); break;
			case 'S': flash_erase(); break;
			case 'P': flash_program(); break;
			case 'V': flash_verify(); break;
			case 'L': flash_protect(); break;
			case 'A': flash_auto(); break;

//			case '\n':
//			case '\r': redraw = 1; break;		// refresh
			}

			setport( USERBITS, 0, 0 );

			if( ch == 'Q' )
			{
				WriteConfig();
				textattr( LIGHTGRAY ); clrscr();
				chdir( oldpath );
				if( buffer ) farfree( (void far *)buffer );
				return( 0 );
			}

			textattr( ( LIGHTGRAY << 4 ) | YELLOW ); clscrn( 11,40, 23,79 );
		}
	}
}
