#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <string.h>
#include <conio.h>
#include <ctype.h>
#include "timer.h"

#define VERSION			"1.0"
#define COPYRIGHT		"Copyright (c)2004 A-Z-E  www.a-z-e.de"

#define CODESIZE		1024
#define DATASIZE		128

#define DO_FILEREAD		1
#define DO_ERASE		2
#define DO_CHECK		4
#define DO_PROGRAM		8
#define DO_VERIFY		16
#define DO_READ			32
#define DO_PROTECT		64
#define DO_FILEWRITE	128

#define PROG_CODE		1
#define PROG_DATA		2
#define PROG_ID			4
#define PROG_CFG		8
#define PROG_MEM_MASK	15

#define PROG_VERIFY		16
#define PROG_CHECK		32
#define PROG_READ		64
#define PROG_PROTECT	128
#define PROG_MODE_MASK	240

//----------------------------
#define PIC_LOAD_CFG		0x00		// (16<-) switch to 0x2000 (0x4000)
#define PIC_INCREMENT		0x06		// next address

#define PIC_LOAD_CODE		0x02		// (16<-)
#define PIC_READ_CODE		0x04		// (->16)

#define PIC_START_PROG		0x08		// some devices ERASE & PROGRAM !

#define PIC_PROG_ONLY		0x18		//
#define PIC_END_PROG		0x0E		//

// Devices with ERASE function
//----------------------------
#define PIC_CHIP_ERASE		0x1F		// Chip Erase TYPE=1

#define PIC_CODE_ERASE		0x09		// Single mem Erase TYPE=1

#define PIC_BULK_SETUP1		0x01		// Erase Setup TYPE=0
#define PIC_BULK_SETUP2		0x07		// ChipErase if ADDR=2007

// Flash chips only
//----------------------------
#define PIC_LOAD_DATA		0x03		// (16<-)
#define PIC_READ_DATA		0x05		// (->16)

#define PIC_DATA_ERASE		0x0B		// works only if not protected

/*
** D0 = DATA out
** D1 = CLOCK
** D2 = LVP
** D3 = MCLR
** D4 = VCC on
**
** ACK= DATA in (DATA out must be H)
*/

#define DATA	1
#define CLOCK	2
#define PGM		4
#define MCLR	8
#define VCC		16

int pinstate, addr, PORT = 0x378;
int buffer[ 0x2100+DATASIZE ];
char l[81], *pos;
unsigned chks;
FILE *x;

void pin( int which, int state )
{
	if( state )
		pinstate |= which;
	else
		pinstate &= ~which;

	outp( PORT, pinstate );
}

/******************************************
** progmode()
**
** put PIC into LVP programming mode
**
*******************************************/
void power( int state )
{
	outp( PORT, pinstate = VCC );			// VCC off, all pins L
	delay( 100 );

	if( state )
	{
		pin( VCC, 0 );						// Vcc on
		us_delay( 5 );
		pin( PGM, 1 );						// activate LVP mode
		us_delay( 5 );
		pin( MCLR, 1 );						// remove RESET
	}
	delay( 100 );
}

void command( int cmd )
{
	int i;

	for( i = 0; i < 6; ++i )
	{
		pin( CLOCK, 1 ); us_delay( 150 );
		pin( DATA, cmd & 1 ); us_delay( 150 );
		cmd >>= 1;
		pin( CLOCK, 0 ); us_delay( 150 );
	}
	us_delay( 250 );
}

void writedata( int d )
{
	int i;
	
	d = ( d & 0x3FFF ) << 1;

	for( i = 0; i < 16; ++i )
	{
		pin( CLOCK, 1 ); us_delay( 150 );
		pin( DATA, d & 1 ); us_delay( 150 );
		d >>= 1;
		pin( CLOCK, 0 ); us_delay( 150 );
	}
	us_delay( 250 );
}

int readdata( void )
{
	int i;
	unsigned d = 0;

	pin( DATA, 1 ); us_delay( 150 );		// make DATA readable

	for( i = 0; i < 16; ++i )
	{
		pin( CLOCK, 1 ); us_delay( 150 );
		if( ( inp( PORT+1 ) & 0x40 ) != 0 )	// ACK
			d |= 0x8000;
		us_delay( 150 );
		d >>= 1;
		pin( CLOCK, 0 ); us_delay( 150 );
	}
	pin( DATA, 0 ); us_delay( 250 );
	return d & 0x3FFF;
}

void do_load( int cmd, int d )
{
	us_delay( 250 );
	command( cmd );
	writedata( d );
}

void increment( int steps )
{
	while( steps-- )
	{
		command( PIC_INCREMENT );
		++addr;
		us_delay( 500 );
	}
	if( ( addr & 0xFF ) ==  0 )
		us_delay( 500 );
}

void progdelay( void )
{
	us_delay( 20000 );				// should take max. 8ms + 5ms
}

int program( int mode )
{
	int retry;

	power( 1 );

	if( mode & PROG_CODE )			// PROGRAM memory space
	{
		printf( "\n" );
		for( addr = 0; addr < CODESIZE; )
		{
			retry = 5;
RETRY1:
			do_load( PIC_LOAD_CODE, buffer[addr] );
			command( PIC_START_PROG );
			progdelay();
			printf( "\r%04X PROGRAM memory", addr );
	
			if( mode & PROG_VERIFY )
			{
				command( PIC_READ_CODE );
				if( readdata() != buffer[addr] )
				{
					if( --retry )
					{
						progdelay();
						goto RETRY1;
					}
					break;
				}
			}

			increment( 1 );
		}
		if( addr != CODESIZE )
			goto ENDPROG;
		mode &= ~PROG_CODE;
	}

	if( mode & PROG_DATA )			// DATA memory space
	{
		do_load( PIC_LOAD_DATA, buffer[0x2100] & 0xFF );
		progdelay();

		printf( "\n" );
		for( addr = 0x2100; addr < 0x2100+DATASIZE; )
		{
			retry = 5;
RETRY2:
			do_load( PIC_LOAD_DATA, buffer[addr] & 0xFF );
			command( PIC_START_PROG );
			progdelay();
			printf( "\r%04X EEPROM memory", addr );

			if( mode & PROG_VERIFY )
			{
				command( PIC_READ_DATA );
				if( ( readdata() & 0xFF ) != ( buffer[addr] & 0xFF ) )
				{
					if( --retry )
					{
						progdelay();
						goto RETRY2;
					}
					break;
				}
			}

			increment( 1 );
		}
		if( addr != 0x2100+DATASIZE )
			goto ENDPROG;
		mode &= ~PROG_DATA;
	}
	if( mode & PROG_ID )			// USER ID memory space
	{
		do_load( PIC_LOAD_CFG, 0 );
		progdelay();
		printf( "\n" );

		for( addr = 0x2000; addr < 0x2004; )
		{
			retry = 5;
RETRY3:
			do_load( PIC_LOAD_CODE, buffer[addr] );
			command( PIC_START_PROG );
			progdelay();
			printf( "\r%04X USER ID", addr );

			if( mode & PROG_VERIFY )
			{
				command( PIC_READ_CODE );
				if( readdata() != buffer[addr] )
				{
					if( --retry )
					{
						progdelay();
						goto RETRY3;
					}
					break;
				}
			}

			increment( 1 );
		}
		if( addr != 0x2004 )
			goto ENDPROG;
		mode &= ~PROG_ID;
	}

	if( mode & PROG_CFG )			// CONFIG memory space
	{
		int val, val2;

		printf( "\n" );
		if( addr < 0x2000 || addr > 0x2007 )
		{
			do_load( PIC_LOAD_CFG, 0 ); addr = 0x2000;
			progdelay();
		}
		while( addr != 0x2007 )
			increment( 1 );

		retry = 5;
RETRY4:
		val = buffer[addr];
		if( mode & PROG_PROTECT )
			val &= ~0x3D00;			// complete CODE & DATA protection

		do_load( PIC_LOAD_CODE, val );
		command( PIC_START_PROG );
		progdelay();
		printf( "\r%04X CONFIG WORD", addr );

		if( mode & PROG_VERIFY )
		{
			power( 1 );
			do_load( PIC_LOAD_CFG, 0 ); addr = 0x2000;
			increment( 7 );

			command( PIC_READ_CODE );
			val2 = readdata();

			if( val2 != val )
			{
				if( --retry )
				{
					progdelay();
					goto RETRY4;
				}
				goto ENDPROG;
			}
		}
		mode &= ~PROG_CFG;
	}

ENDPROG:
	power( 0 );
	return mode;
}

int read_verify( int mode )
{
	int val;

	power( 1 );

	if( mode & PROG_CODE )			// PROGRAM memory space
	{
		printf( "\n" );
		for( addr = 0; addr < CODESIZE; )
		{
			command( PIC_READ_CODE );
			printf( "\r%04X PROGRAM memory", addr );

			val = readdata();
			if( mode & PROG_READ )
				buffer[addr] = val;
			else if( ( ( mode & PROG_VERIFY ) && val != buffer[addr] ) ||
					 ( ( mode & PROG_CHECK ) && val != 0x3FFF ) )
			{
				power(0);
				return 0;
			}

			increment( 1 );
		}
	}

	if( mode & PROG_DATA )			// DATA memory space
	{
		printf( "\n" );
		for( addr = 0x2100; addr < 0x2100+DATASIZE; )
		{
			command( PIC_READ_DATA );
			printf( "\r%04X EEPROM memory", addr );

			val = readdata() & 0xFF;
			if( mode & PROG_READ )
				buffer[addr] = val;
			else if( ( ( mode & PROG_VERIFY ) && val != buffer[addr] ) ||
					 ( ( mode & PROG_CHECK ) && val != 0xFF ) )
			{
				power(0);
				return 0;
			}
			
			increment( 1 );
		}
	}

	if( mode & PROG_ID )			// USER ID memory space
	{
		printf( "\n" );
		do_load( PIC_LOAD_CFG, 0 );
		for( addr = 0x2000; addr < 0x2004; )
		{
			command( PIC_READ_CODE );
			printf( "\r%04X USER ID", addr );
			
			val = readdata();
			if( mode & PROG_READ )
				buffer[addr] = val;
			else if( ( ( mode & PROG_VERIFY ) && val != buffer[addr] ) ||
					 ( ( mode & PROG_CHECK ) && val != 0x3FFF ) )
			{
				power(0);
				return 0;
			}

			increment( 1 );
		}
	}

	if( mode & PROG_CFG )			// CONFIG memory space
	{
		printf( "\n" );
		if( addr < 0x2000 || addr > 0x2007 )
		{
			do_load( PIC_LOAD_CFG, 0 );
			addr = 0x2000;
		}
		while( addr != 0x2007 )
			increment( 1 );

		command( PIC_READ_CODE );
		printf( "\r%04X CONFIG WORD", addr );

		val = readdata();
		if( mode & PROG_READ )
			buffer[addr] = val;
		else if( ( ( mode & PROG_VERIFY ) && val != buffer[addr] ) ||
					( ( mode & PROG_CHECK ) && val != 0x3FFF ) )
		{
			power(0);
			return 0;
		}
	}

	power( 0 );
	return 1;
}

void erase( void )
{
	power( 1 );

	do_load( PIC_LOAD_CFG, 0x3FFF );
	increment( 7 );				// advance to 0x2007 = CFG WORD
	command( PIC_BULK_SETUP1 );
	command( PIC_BULK_SETUP2 );
	command( PIC_START_PROG );
	progdelay();
	command( PIC_BULK_SETUP1 );
	command( PIC_BULK_SETUP2 );
	us_delay( 100 );

	power( 0 );
	us_delay( 10000 );
}

int getbyte( void )
{
	int d1, d2;

	d1 = *pos++ - '0';
	if( d1 > 9 )
		d1 -= 7;
	d2 = *pos++ - '0';
	if( d2 > 9 )
		d2 -= 7;

	d1 = ( d1 << 4 ) | d2;
	chks += d1;
	return d1;
}

int readhex( char *name )
{
	int len, mode, val;

	if( ( x = fopen( name, "r" ) ) == NULL )
		return 0;

	while( fgets( l, 80, x ) != NULL )
	{
		if( l[0] != ':' )
			continue;

		pos = l+1;

		chks = 0;

		len = getbyte();
		
		addr = getbyte() * 256;
		addr |= getbyte();
		
		mode = getbyte();
		if( mode != 0 )
			continue;

		while( len-- )
		{
			val = getbyte();

			if( addr < 0x4200 )
			{
				if( addr & 1 )				// ODD (high) byte
				{
					buffer[ addr / 2 ] &= 0xFF;
					buffer[ addr / 2 ] |= ( val & 0x3F ) * 256;
				}
				else
				{
					buffer[ addr / 2 ] &= 0xFF00;
					buffer[ addr / 2 ] |= val;
				}
			}
			else
				buffer[ addr - 0x2100 ] = val;

			++addr;
		}

		getbyte();
		if( ( chks & 0xFF ) != 0 )
		{
			fclose( x );
			return 0;
		}
	}

	buffer[0x2007] |= 0x0280;		// CONFIG WORD reserved bit & LVP bit

	fclose( x );
	return 1;
}

int writehex( char *name )
{
	int i;

	if( ( x = fopen( name, "w" ) ) == NULL )
		return 0;

	// PROGRAM memory
	for( addr = 0; addr < CODESIZE; addr += 8 )
	{
		fprintf( x, ":10%04X00", addr*2 );
		
		chks = 0x10 + ((addr*2) >> 8) + ((addr*2) & 0xFF);
		for( i = 0; i < 8; ++i )
		{
			fprintf( x, "%02X", buffer[addr+i] & 0xFF );
			fprintf( x, "%02X", ( buffer[addr+i] >> 8 ) & 0xFF );
			chks += buffer[addr+i] & 0xFF;
			chks += ( buffer[addr+i] >> 8 ) & 0xFF;
		}
		fprintf( x, "%02X\n", ( 0x100 - ( chks & 0xFF ) ) & 0xFF );
	}

	// DATA memory
	for( addr = 0x2100; addr < 0x2100+DATASIZE; addr += 16 )
	{
		fprintf( x, ":10%04X00", addr*2 );

		chks = 0x10 + ((addr*2) >> 8) + ((addr*2) & 0xFF);
		for( i = 0; i < 16; ++i )
		{
			fprintf( x, "%02X", buffer[addr+i] & 0xFF );
			chks += buffer[addr+i] & 0xFF;
		}
		fprintf( x, "%02X\n", ( 0x100 - ( chks & 0xFF ) ) & 0xFF );
	}

	// USER ID
	addr = 0x2000;
	fprintf( x, ":08400000" );

	chks = 0x48;
	for( i = 0; i < 4; ++i )
	{
		fprintf( x, "%02X", buffer[addr+i] & 0xFF );
		fprintf( x, "%02X", ( buffer[addr+i] >> 8 ) & 0xFF );
		chks += buffer[addr+i] & 0xFF;
		chks += ( buffer[addr+i] >> 8 ) & 0xFF;
	}
	fprintf( x, "%02X\n", ( 0x100 - ( chks & 0xFF ) ) & 0xFF );

	// CHIP ID & CONFIGURATION WORD
	addr = 0x2006;
	fprintf( x, ":02400C00" );

	chks = 0x4E;
	for( i = 0; i < 2; ++i )
	{
		fprintf( x, "%02X", buffer[addr+i] & 0xFF );
		fprintf( x, "%02X", ( buffer[addr+i] >> 8 ) & 0xFF );
		chks += buffer[addr+i] & 0xFF;
		chks += ( buffer[addr+i] >> 8 ) & 0xFF;
	}
	fprintf( x, "%02X\n", ( 0x100 - ( chks & 0xFF ) ) & 0xFF );

	fprintf( x, ":00000001FF\n" );

	fclose( x );
	return 1;
}

void Header( void )
{
	puts( "\nPIC16 Version " VERSION "  " COPYRIGHT "\n" );
}

volatile void Usage( void )
{
	puts(	"Usage: PIC16 {cmds} [-arg]* <hexfile> [<hexfile2>]\n"
			"  where cmds are:\n"
			" E  ERASE             B  BLANKCHECK\n"
			" P  PROGRAM           V  VERIFY pic against buffer\n"
			" L  LOCK (PROTECT)    R  READ pic into <hexfile> or <hexfile2>\n"
			" A  ERASE + BLANKCHECK + PROGRAM + VERIFY + LOCK\n"
			"  and args are:\n"
			" -Px select printerport x=0:PRN 1=LPT1 2=LPT2 (default=LPT1)\n"
			" -C  exclude CODE memory from reading/programming\n"
			" -D  exclude DATA memory from reading/programming\n"
			" -I  exclude USER ID memory from reading/programming\n"
			" -X  exclude CONFIG word from reading/programming\n"
			" -V  don't verify each word after writing\n"
			"\n"
			" Example:\n"
			" PIC16 A file.hex     Erase, program file1.hex, verify, protect\n"
			" PIC16 V -D file.hex  Verify PIC against file.hex, ignoring DATA memory\n"
			" PIC16 R -X file.hex  Read PIC into file.hex, leave CONFIG WORD out\n"
			" PIC16 EB             ERASE and BLANKCHECK pic\n"
		);
	exit( -1 );
}

volatile void leave( void )
{
	puts( "Press any key" );
	if( getch() == 0 )
		getch();
	exit( -1 );
}

int main( int ac, char *av[] )
{
	char *fname = NULL, *fname2 = NULL, *cmd = NULL;
	int val;
	int pmode, mode = 0;
	int progmode = PROG_CODE | PROG_DATA | PROG_ID | PROG_CFG | PROG_VERIFY;

	delay(0);

	for( addr = 0; addr < 0x2000; ++addr )
		buffer[addr] = 0x3FFF;
	for( addr = 0x2100; addr < 0x2200; ++addr )
		buffer[addr] = 0x00FF;
	for( addr = 0x2000; addr < 0x2004; ++addr )
		buffer[addr] = 0x3FFF;
	for( addr = 0x2004; addr < 0x2006; ++addr )
		buffer[addr] = 0;
	buffer[0x2006] = 0x07C0;
	buffer[0x2007] = 0x3FFF;

	power(0);

	while( --ac )
	{
		pos = *++av;

		if( pos[0] == '/' || pos[0] == '-' )
		{
			switch( toupper( pos[1] ) )
			{
			case 'P':
				switch( pos[2] )
				{
				case '0': PORT = 0x3BC; break;
				case '1': PORT = 0x378; break;
				case '2': PORT = 0x278; break;
				default : puts( "Unknown port number" ); Usage();
				}
				break;

			case 'D':		// exclude DATA from programming
				progmode &= ~PROG_DATA;
				break;

			case 'I':		// exclude ID from programming
				progmode &= ~PROG_ID;
				break;

			case 'C':		// exclude CODE from programming
				progmode &= ~PROG_CODE;
				break;

			case 'X':		// exclude CONFIGWORD from programming
				progmode &= ~PROG_CFG;
				break;

			case 'V':		// no inline verify
				progmode &= ~PROG_VERIFY;
				break;

			default : puts( "Unknown switch" );Usage();
			}
		}
		else if( cmd == NULL )
			cmd = pos;
		else if( fname == NULL )
			fname = pos;
		else if( fname2 == NULL )
			fname2 = pos;
		else
		{
			puts( "Too many arguments" );
			Usage();
		}
	}

	while( *cmd )
	{
		switch( toupper( *cmd ) )
		{
		case 'E': mode |= DO_ERASE; break;
		case 'B': mode |= DO_CHECK; break;
		case 'P': mode |= DO_PROGRAM|DO_FILEREAD; break;
		case 'V': mode |= DO_VERIFY|DO_FILEREAD; break;
		case 'R': mode |= DO_READ|DO_FILEWRITE; break;
		case 'L': mode |= DO_PROTECT; break;
		case 'A': mode |= DO_ERASE|DO_CHECK|DO_FILEREAD|DO_PROGRAM|DO_VERIFY|DO_PROTECT; break;
		default : printf( "Unknown command '%c'\n", *cmd ); Usage();
		}
		++cmd;
	}

	if( !fname && ( mode & (DO_FILEREAD|DO_FILEWRITE) ) != 0 )
	{
		puts( "Must specify a filename" );
		Usage();
	}
	if( !fname2 && ( mode & (DO_FILEREAD|DO_FILEWRITE) ) == (DO_FILEREAD|DO_FILEWRITE) )
	{
		puts( "Must specify 2 filenames" );
		Usage();
	}

	Header();

	printf( "Insert PIC and press any key (ESC to abort)\n" );
	switch( getch() )
	{
	case 0: getch(); break;
	case 0x1B:
		return 1;
	}

 	if( mode & DO_FILEREAD )
	{
		printf( "\rReading '%s'...\n", fname );
		if( !readhex( fname ) )
		{
			printf( " - ERROR! Aborting...\n" );
			leave();
		}
	}

	power(1);
	do_load( PIC_LOAD_CFG, 0 );
	increment( 7 );
	command( PIC_READ_CODE );
	val = readdata();
	power(0);

	if( ( val & 0x3D00 ) != 0x3D00 &&		// some protection active ?
		( mode & DO_PROGRAM ) != 0 &&		// we want to write ?
		( mode & DO_ERASE ) == 0 )			// and erase not activated ?
	{
		printf( "\rChip is protected! Erasing...\n" );
		erase();
	}
	else if( mode & DO_ERASE )
	{
		printf( "\rErasing...\n" );
		erase();
	}

	if( mode & DO_CHECK )
	{
		printf( "\rBlank Checking..." );

		if( !read_verify( ( progmode & PROG_MEM_MASK ) | PROG_CHECK ) )
		{
			printf( " - Not empty!\n", addr );
			leave();
		}
	}

	if( mode & DO_PROGRAM )
	{
		int retries = 5;

		printf( "\rProgramming..." );
		pmode = (progmode & ~PROG_PROTECT) | ( (mode & DO_VERIFY) ? 0 : PROG_VERIFY );
		while( ( ( pmode = program( pmode ) ) & PROG_MEM_MASK ) != 0 )
		{
			if( !--retries )
			{
				printf( " - ERROR! Aborting...\n", addr );
				leave();
			}
		}
	}
	 
	if( mode & DO_VERIFY )
	{
		printf( "\rVerifying..." );
		if( !read_verify( ( progmode & PROG_MEM_MASK ) | PROG_VERIFY ) )
		{
			printf( " - ERROR! Aborting...\n", addr );
			leave();
		}
	}

	if( mode & DO_READ )
	{
		printf( "\rReading..." );
		read_verify( ( progmode & PROG_MEM_MASK ) | PROG_READ );
	}

	if( mode & DO_PROTECT )
	{
		int retries = 3;

		printf( "\rProtecting..." );
		pmode = PROG_CFG | PROG_PROTECT | PROG_VERIFY;
		while( ( ( pmode = program( pmode ) ) & PROG_MEM_MASK ) != 0 )
		{
			if( !--retries )
			{
				printf( " - ERROR! Aborting...\n", addr );
				leave();
			}
		}
	}

	if( mode & DO_FILEWRITE )
	{
		if( ( mode & DO_FILEREAD ) != 0 && fname2 )
			fname = fname2;

		printf( "\rWriting to file '%s'...\n", fname );
		if( !writehex( fname ) )
		{
			printf( " - ERROR! Aborting...\n" );
			leave();
		}
	}
	
	printf( "\nDONE - BYE!\n" );
	return 0;
}
