/*

    chksum.c

    (c) 2012  -  Version 1.0

    chksum - Programa que calcula el valor Checksum, Checksum 32, CRC-CCITT,
    CRC-16, CRC-32 y Fluke signature (usado en el Fluke 9010a) de un fichero
    dado como argumento al programa.



    Para compilarlo por lnea de comandos:


    bcc -1- -G -O2 -c -mt -v- chksum.c
    tlink /Tdc lib\c0t chksum, chksum,, lib\cs

    Genera un fichero COM, optimizado para mxima rapidez, y sin informacin
    de depuracin compatible con el procesador 8088 y superiores.


    bcc -1- -G -O2 -c -ms -v- chksum.c
    tlink /Tde lib\c0s chksum, chksum,, lib\cs

    Genera un fichero EXE, optimizado para mxima rapidez, y sin informacin
    de depuracin compatible con el procesador 8088 y superiores.


    d:\msys32new>new-vars
    d:\msys32new>gcc -O2 chksum.c -o chksum.exe
    d:\msys32new>strip chksum.exe
    d:\msys32new>copy chksum.exe c:\hxbeta\bin
    c:\hxbeta\bin>pestub chksum.exe (ahora chksum.exe necesita dkrnl32.dll,
                                     msvcrt.dll, dpmild32.exe y hdpmi32.exe
                                     para su funcionamiento)

    Genera un fichero EXE, optimizado para mxima rapidez, sin informacin
    de depuracin y compatible con DOS, Windows 98, Windows XP, Windows 7 y
    Windows 10



    Para compilarlo desde el entorno integrado:

    Abrimos un proyecto, seleccionando las opciones de optimizacin y de
    depuracin que nos interesen, generando un fichero tipo EXE.
    Despus, en la lnea de comandos del DOS:

    exe2bin chksum.exe chksum.com

*/



#include <stdio.h>
#include <stdlib.h>

#define CRCCCITT 0x1021                                 /* polinomio CCITT */
#define CRCCCITT_REV 0x8408                     /* polinomio CCITT inverso */
#define CRC16 0x8005                                    /* polinomio CRC16 */
#define CRC16_REV 0xA001                        /* polinomio CRC16 inverso */

#define crcupdate(d, a, t)        *(a) = (*(a) << 8) ^ (t)[(*(a) >> 8) ^ (d)]
#define crcrevupdate(d, a, t)         *(a) = (*(a) >> 8) ^ (t)[(*(a) ^ (d)) \
					      & 0x00FF]
unsigned long crc_table[256];



unsigned short crchware(unsigned short data, unsigned short genpoly,
			unsigned short accum)
{
    int i;

    data <<= 8;                                      /* datos al byte alto */
    for (i = 8; i > 0; i--)
	{
	    if ((data ^ accum) & 0x8000)
				     /* si b+s de (data XOR accum) es TRUE */
		accum = (accum << 1) ^ genpoly;
					     /* desplaza y resta polinomio */
	    else
		accum <<= 1;         /* si no, desplazamiento transparente */
	    data <<= 1;	                    /* sube bit siguiente para XOR */
	}
    return accum;
}


unsigned short crcrevhware(unsigned short data, unsigned short genpoly,
			   unsigned short accum)
{
    int i;

    for (i = 8; i > 0; i--)
	{
	    if ((data ^ accum) & 0x0001)/* si b-s de (data XOR accum) TRUE */
		accum = (accum >> 1) ^ genpoly;        /* desplaza y resta */
	    else
		accum >>= 1;         /* si no, desplazamiento transparente */
	    data >>= 1;                     /* baja bit siguiente para XOR */
	}
    return accum;
}


unsigned short * mk_crctbl(unsigned short poly,
			   unsigned short (* crcfn) (unsigned short,
						     unsigned short,
						     unsigned short))
		       /* el segundo parmetro es un puntero a funcin CRC */
{
    unsigned short * crctp;                          /* puntero a la tabla */
    int i;

    if ((crctp = (unsigned short *) malloc(256 * sizeof(unsigned short))) ==
	 NULL)
	return NULL;
    for (i = 0; i < 256; i++)    /* rellena la tabla con CRC de valores... */
	crctp[i] = (*crcfn) (i, poly, 0);                   /* ... 0 - 255 */
    return crctp;                        /* devuelve un puntero a la tabla */
}


void gen_table(void)                /* build the crc table */
{
    unsigned long crc, poly;
    int	i, j;

    poly = 0xEDB88320L;
    for (i = 0; i < 256; i++)
	{
	crc = i;
	for (j = 8; j > 0; j--)
	    {
	    if (crc & 1)
		crc = (crc >> 1) ^ poly;
	    else
		crc >>= 1;
	    }
	crc_table[i] = crc;
	}
}


unsigned short int ShiftBits(unsigned long sig)
{
  return ((sig >> 6) ^ (sig >> 8) ^ (sig >> 11) ^ (sig >> 15));
}



int main(int argc, char *argv[])
{
    FILE * handle;
    unsigned short * tablep;
    unsigned short * tablep2;
    int c;
    int i, b, tmp, p, n, bits;
    int crcData[16];
    int pointers[5];
    unsigned short chksum = 0;
    unsigned long chksum32 = 0;
    unsigned short crcccitt = 0;
    unsigned short crc16 = 0;
    unsigned long crc32 = 0xFFFFFFFF;
    unsigned long flukesig = 0;


    if (argc < 2)
	exit(puts("\nUsage:\nchksum filename.ext"));
    if ((handle = fopen(argv[1], "rb")) == NULL)
	exit(printf("\nCannot open input file %s\n", argv[1]));
    if ((tablep2 = mk_crctbl(CRCCCITT, crchware)) == NULL)
        exit(puts("\nNot enough memory for CRC-CCITT table."));
    if ((tablep = mk_crctbl(CRC16_REV, crcrevhware)) == NULL)
	exit(puts("\nNot enough memory for CRC-16 table."));

    gen_table();                         /* Used in CRC-32 calculation */

    for (i = 0; i < 16; i++)  /* Used        */
      crcData[i] = 0;         /* in          */
    pointers[0] = 6;          /* calculation */
    pointers[1] = 8;          /* of          */
    pointers[2] = 11;         /* Fluke       */
    pointers[3] = 15;         /* signature   */
    pointers[4] = 0;

    while ((c = fgetc(handle)) != EOF)
      {
	tmp = c;

	chksum += c;                     /* Calculates Checksum */

        chksum32 += c;                   /* Calculates Checksum 32 */

        crcupdate(c, &crcccitt, tablep2);/* Used in CRC-CCITT calculation */

	crcrevupdate(c, &crc16, tablep); /* Used in CRC-16 calculation */

	crc32 = (crc32 >> 8) ^ crc_table[(crc32 ^ c) & 0xFF];
					 /* Used in CRC-32 calculation */

	for (i = 0; i < 5; i++)          /* Used        */
	  {
	    p = pointers[i];             /* in          */
	    if (i < 4)
	      tmp = tmp ^ crcData[p];    /* calculation */
	    else
	      crcData[p] = tmp;          /* of          */
	    --p;
	    if (p < 0)                   /* Fluke       */
	      p = 15;
	    pointers[i] = p;             /* signature   */
	  }
      }

    crc32 ^= 0xFFFFFFFF;              /* Used in CRC-32 calculation */

    for (n = 0; n < 16; n++)                     /* Used            */
      {
	tmp = crcData[n];                        /* in              */
	for (i = 0; i < 8; ++i)
	  {                                      /* calculation     */
	    bits = tmp ^ (ShiftBits(flukesig));
	    flukesig <<= 1;                      /* of              */
	    flukesig |= (bits & 0x01);
	    tmp >>= 1;                           /* Fluke signature */
	  }
      }
    flukesig &= 0xFFFF;   /* Used in calculation of Fluke signature */

    printf("\nChecksum = %04X\n", chksum);
    printf("Checksum 32 = %08lX\n", chksum32);
    printf("CRC - CCITT = %04X\n", crcccitt);
    printf("CRC - 16 = %04X\n", crc16);
    printf("CRC - 32 = %08lX\n", crc32);
    printf("Fluke signature = %04X\n", flukesig);
    free(tablep2);
    free(tablep);
    fclose(handle);
    return 0;
}
