* APPENDIX A - IO Polling to eeprom application source


RAM  		EQU  	$50        	; Ram space starts at $50
ROM  		EQU  	$100       	; program space starts at $0100
VECTORS  	EQU  	$1ff4  	; Reset+ interrupt vectors start at $1ff4


******************************************
* Eeprom type and configuration switches
*
#SETNOT 	9346FORM8   	; 9346 eeprom, 1 byte format.
#SETNOT 	9346FORM16  	; 9346, 2 byte word format.
#SETNOT 	9356FORM16  	; 9356, 2 byte word format.
#SET 		9366FORM16  	; 9366, 2 byte word format.

#SET 		AUTOERASE  		; For eeproms that do not need to
                			; erase before writing.
#SET 		AUTOSEQ    		; For eeproms that automatically
                			; sequence to the next cell when
                			; being read.
*
******************************************
*
* RAM - variables
*
*
*

    	ORG  RAM

ee_addr     ds  	1 		; eeprom address stored here.
mem_addr    ds  	1 		; Block index stored here.
block_to_go ds  	1 		; Block length stored here.

data  	ds  	$ad    	; Rest of data space is data to be stored.

*
* PROGRAM
*
* The main subroutines are READ, EWEN, EWDS,
* WRITE, WRAL, ERASE, and ERAL.  SLACK, WAIT,
* CLOCK, and SHUFFLE support
* these.
*
* Port locations follow.
*
*

DIPORT  	EQU	2 		; Eeprom "DI."
DILINE  	EQU   6 		; portc.6, an output line.

DOPORT  	EQU   2 		; Eeprom "DO."
DOLINE  	EQU   7 		; portc.7, an input line.

CLKPORT  	EQU   2 		; Eeprom CLocK.
CLKLINE  	EQU   5 		; portc.5, an output line.

CSPORT   	EQU  	0 		; Eeprom Chip Select.
CSLINE   	EQU  	0 		; porta.0, an output line.

    	
	ORG	ROM

*********************************************;
* WAIT - This routine delays the next command
*        to the eeprom until the most recent
*        write or erase has finished.
*        If in a write or erase
*        cycle the routine loops.  One
*        write or erase takes 4
*        milliseconds.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - nothing
*
WAIT:
     		bset 	CSLINE,CSPORT  	; Select.

     		brclr DOLINE,DOPORT,$  	; Loop here until eeprom ready.
     		bclr 	CSLINE,CSPORT  	; De-select.
     		rts

*********************************************;
* CLOCK# - clock data in to or out of the eeprom
*          using the # number of clocks.
*          "D_CARE" is used to handle the
*          'don't care' clocks required of
*          some commands.  It is conditionally
*          defined.  The required number of
*          'don't care' clocks is a function of
*          eeprom type and form.
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - nothing
*

CLOCK6:  					; Clocks six clocks to the eeprom.
#IF 		9366FORM16
D_CARE:  					; 9366 Form 16 uses 6 don't care 	
					; bits.
#ENDIF

#IF 9356FORM16
D_CARE:  					; 9356 Form 16 uses 6 don't care 	
					; bits.
#ENDIF
     		bset 	CLKLINE,CLKPORT	; Active clock.
     		bclr 	CLKLINE,CLKPORT  	; Inactive clock.

#IF 		9346FORM8
D_CARE:  					; 9346 Form 8 uses 5 don't care 	
					; bits.
#ENDIF

CLOCK5:  					; Clocks five clocks to the eeprom.
     		bset 	CLKLINE,CLKPORT  	; Active clock.
     		bclr 	CLKLINE,CLKPORT  	; Inactive clock.

#IF 		9346FORM16
D_CARE:  					; 9346 Form 16 uses 4 don't care 	
					; bits.
#ENDIF

CLOCK4:  					; Clocks four clocks to the eeprom.
     		bset 	CLKLINE,CLKPORT  	; Active clock.
     		bclr 	CLKLINE,CLKPORT  	; Inactive clock.
CLOCK3:  					; Clocks three clocks to the eeprom.
     		bset 	CLKLINE,CLKPORT  	; Active clock.
     		bclr 	CLKLINE,CLKPORT  	; Inactive clock.
CLOCK2:  					; Clocks two clocks to the eeprom.
     		bset 	CLKLINE,CLKPORT  	; Active clock.
     		bclr 	CLKLINE,CLKPORT  	; Inactive clock.
CLOCK:   					; Clocks one clock to the eeprom.
     		bset 	CLKLINE,CLKPORT  	; Active clock.
     		bclr 	CLKLINE,CLKPORT  	; Inactive clock.
     		rts

*********************************************;
* EESEND    - sends the complement of the carry
*             to the eeprom and rotates the
*             accumulator left through the carry.
*
* INPUTS    - accumulator
* OUTPUTS   - accumulator left rotated through
*             carry, and one bit to the Eeprom.
* DESTROYS  - nothing
*
EESEND:
     		bcc  	OPU1           	; If carry clear jump to set.
     		bclr 	DILINE,DIPORT  	; If carry set clear the output to
						; eeprom.
     		bra  	OPU0
OPU1:
     		bset 	DILINE,DIPORT  	; Clear carry means set output to 	
					; eeprom.
OPU0:  					; Clock the complement of the carry 
						; into the eeprom.
     		bset 	CLKLINE,CLKPORT  	; Active clock.
     		bclr 	CLKLINE,CLKPORT  	; Inactive clock.
     		rola   			; ready the next bit to be sent by 	
					; rotating to carry.
     		rts

*********************************************;
* SENDADR 	- Send a 6,7, or 8 bit address to
*             the serial eeprom depending on its
*             type and form.
*  -or-
* SENDDAT 	- Send 8 bits of data.
*
* INPUTS    - Byte address in accumulator.
*             In 16 bit format bit 0 is ignored.
* OUTPUTS   - none
* DESTROYS  - Accumulator
*
SENDDAT:
     		rola   			; ready the first data bit to be 	
					; sent by rotating to carry.
     		jsr  	EESEND  		; Send data bit 7 or 15.
     		jsr  	EESEND  		; Send data bit 6 or 14.
     		bra  	RTTZ
SENDADR:
     		coma    			; Addresses are inverted twice 	
					; before being sent!
#IF 		9346FORM8
     		rola   			; Rotate address extra bit through 	
					; carry.
     		rola   			; ready the first address bit to be 
						; sent by rotating to carry.
     		jsr  	EESEND  		; Send address bit 6.
#ENDIF

#IF 		9346FORM16
     		rola   			; Rotate extra address bit through 	
					; carry.
     		rola   			; Rotate extra address bit through 	
					; carry.
     		rola   			; ready the first address bit to be 
						; sent by rotating to carry.
#ENDIF

#IF 		9356FORM16
     		ora  	#$80      		; Set the Don't care bit.
     		rola           		; Rotate the Don't care bit to the 	
					; carry.
     		jsr  	EESEND    		; Send 1 Don't care bit.
     		jsr  	EESEND    		; Send address bit 6.
#ENDIF

#IF 		9366FORM16
     		rola   			; ready the first address bit to be 
						; sent by rotating to carry.
     		jsr  	EESEND    		; Send address bit 7.
     		jsr  	EESEND    		; Send address bit 6.
#ENDIF

RTTZ:
     		jsr  	EESEND  		; Send bit 5 or 13.
     		jsr  	EESEND  		; Send bit 4 or 12.
     		jsr  	EESEND  		; Send bit 3 or 11.
     		jsr  	EESEND  		; Send bit 2 or 10.
     		jsr  	EESEND  		; Send bit 1 or 9.
     		jsr  	EESEND  		; Send bit 0 or 8.
     		rts

*********************************************;
* SUEEP - Set up the eeprom ports.  Called
*         frequently to ensure the ports are
*         set up for the eeprom and so that
*         other tasks can share the ports.
*
* INPUTS   - none
* OUTPUTS  - DDRA,DDRB
* DESTROYS - nothing
*
SUEEP:
     		bset 	CSLINE,CSPORT+4 	; Chip Select port is
     		bclr 	CSLINE,CSPORT   	; output and low.
     		bset 	CLKLINE,CLKPORT+4 ; Clock is output.
     		bclr 	DOLINE,DOPORT+4   ; DO is an input.
     		bset 	DILINE,DIPORT+4   ; DI is an output.
     		rts

*********************************************;
* EWEN - This subroutine enables erase and write
*        operations.  It in effect unlocks the
*        eeprom so that its cells may be
*        changed.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - nothing
*
EWEN:
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		bset 	DILINE,DIPORT 	; Send 1.
     		jsr  	CLOCK         	; Clock it into the eeprom.
     		bclr 	DILINE,DIPORT 	; Send 00.
     		jsr  	CLOCK2        	; Clock them into the eeprom.
     		bset 	DILINE,DIPORT 	; Send 11.
     		jsr  	CLOCK2        	; Clock them into the eeprom.
     		bclr 	DILINE,DIPORT 	; DI line low.
     		jsr  	D_CARE        	; Clock the Don't care clocks.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		rts

*********************************************;
* EWDS - This subroutine disables erase and
*        write operations so that data cannot be
*        inadvertently corrupted.  It in effect
*        locks the eeprom so that its cells
*        cannot be changed.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - nothing
*
EWDS:
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
   	  	bset 	DILINE,DIPORT 	; Send 1.
     		jsr  	CLOCK         	; Clock it into the eeprom.
     		bclr 	DILINE,DIPORT 	; Send 0000.
     		jsr  	CLOCK4        	; Clock them into the eeprom.
     		bclr 	DILINE,DIPORT 	; DI line low.
     		jsr  	D_CARE        	; Clock the Don't care clocks.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		rts

*********************************************;
* ERAL - This subroutine erases the entire
*        eeprom.  An erased cell will put a high
*        level on the DO line when read, but
*        due to inverting in READ, the result
*        will arrive as 0x00 in 6805 memory.
*        ERAL calls EWEN to allow erasure.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - all contents of eeprom
*
ERAL:
     		jsr  	SUEEP        	; Set up the ports for the eeprom.
     		jsr  	EWEN         	; Open the eeprom for writing.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		bset 	DILINE,DIPORT 	; Send 1.
     		jsr  	CLOCK         	; Clock it into the eeprom.
     		bclr 	DILINE,DIPORT 	; Send 00.
     		jsr  	CLOCK2        	; Clock them into the eeprom.
     		bset 	DILINE,DIPORT 	; Send 1.
     		jsr  	CLOCK         	; Clock it into the eeprom.
     		bclr 	DILINE,DIPORT 	; Send 0.
     		jsr  	CLOCK         	; Clock it into the eeprom.
     		jsr  	D_CARE        	; Clock the Don't care clocks.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		jsr  	WAIT          	; Pause until the eeprom comes 	
					; ready.
     		rts

*********************************************;
* WRAL - In FORM8 eeproms this subroutine
*        writes the byte in the accumulator to
*        every byte of the Eeprom.  In FORM16
*        eeproms the accumulator is written to
*        the most significant byte the X
*        register is written to the less
*        significant byte.
*
* INPUTS    - Accumulator ( and X for FORM16)
* OUTPUTS   - none
* DESTROYS  - accumulator in FORM16 applications.
*

#MACRO 	WRAL16
     		txa             		; Put more significant byte into X.
     		jsr  	SENDDAT    		; Send that byte to the eeprom for
                     			; writing.
#MACROEND

WRAL:
     		jsr  	SUEEP        	; Set up the ports for the eeprom.

#IFNOT 	AUTOERASE
     		jsr  	ERAL         	; Erase and open the eeprom.
#ELSEIF
     		jsr  	EWEN         	; Open the eeprom for writing.
#ENDIF
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		bset 	DILINE,DIPORT 	; Send 1.
     		jsr  	CLOCK         	; Clock it into the eeprom.
     		bclr 	DILINE,DIPORT 	; Send 000.
     		jsr  	CLOCK3        	; Clock them into the eeprom.
     		bset 	DILINE,DIPORT 	; Send 1.
     		jsr  	CLOCK         	; Clock it into the eeprom.
     		bclr 	DILINE,DIPORT 	; Send DI low.
     		jsr  	D_CARE        	; Clock the Don't care clocks.

     		jsr  	SENDDAT       	; Send a byte for writing.

#IF 		9346FORM16
     		WRAL16             	; Send a second byte if form 16.
#ENDIF

#IF 		9356FORM16
     		WRAL16             	; Send a second byte if form 16.
#ENDIF

#IF 		9366FORM16
     		WRAL16             	; Send a second byte if form 16.
#ENDIF
     		bclr	DILINE,DIPORT 	; Send DI line low.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom

     		jsr  	WAIT  		; <- Waits here for erasure to
                				; finish.

     		jsr  	EWDS         	; Close the eeprom for writing.
     		rts

*********************************************;
* ERASE - This subroutine Erases an eight
*         cell byte or 16 cell word in the
*         Eeprom.  The address of the cell is
*         located in the accumulator.  The
*         accumulator is returned unchanged.
*
* INPUTS    - Eeprom address for erasure in Acc.
* OUTPUTS   - none
* DESTROYS  - X.
*

ERASE:
     		tax         		; Store address in X.
     		jsr  	SUEEP        	; Set up the ports for the eeprom.
     		jsr  	EWEN         	; Open the eeprom for writing.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		bset 	DILINE,DIPORT 	; Send 111.
     		jsr  	CLOCK3        	; Clock them into the eeprom.

     		jsr  	SENDADR    		; Send the 6,7, or 8 bit address.

     		bclr 	DILINE,DIPORT 	; DI line low.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom

     		jsr  	WAIT  		; <- Waits here for erasure to
                				; finish.

     		txa    			; Return the address to accumulator.
     		rts

**********************************************
* Write macros ------------------------------

#MACRO 	WRBYTE
     		ldx  	mem_addr  		; Bring in the address pointer
                    			; of the byte to be written.
     		lda  	,x        		; Bring the byte to be written
                    			; into the accumulator.
     		incx           		; Increment the address pointer.
     		stx  	mem_addr  		; Update the address pointer.
     		jsr  	SENDDAT   		; Send accumulator to eeprom for
                    			; writing.
     		lda  	block_to_go 	; load block length.
     		deca         		; Decrement length and check if 	
					; done.
     		sta  	block_to_go 	; Update block length.
#MACROEND

#MACRO 	INIT16
        	lda	block_to_go 	; Check for Zero length.
        	beq  	WRDONE    		; Abort if Zero.
        	rora            		; Place LS bit of address in carry.
        	bcc 	LEN_OK  		; Ensure that block_to_go
        	rola            		; starts as an even number.
        	inca            		; increment if not.
        	sta 	block_to_go		; Update to new even value.
LEN_OK:
#MACROEND

*********************************************;
* WRITE - This subroutine Writes a block of
*         eight cell bytes to the Eeprom.
*         Writing starts at low memory value
*         in both eeprom, ee_addr, and 6805
*         memory, mem_addr, and increments
*         upward as block_to_go is decremented
*         downward.
*
* INPUTS    - The following memory locations
*             set up as follows.
*         ee_addr   -> contains the absolute
*                      address of where the
*                      data will start in
*                      the eeprom.
*         mem_addr  -> contains the absolute
*                      starting address of the
*                      block of memory
*                      to be written to eeprom.
*         block_to_go  -> The length of the block,
*                      1 writes one byte,
*                      0 writes none.
* OUTPUTS   - none
* DESTROYS  - ee_addr, mem_addr, block_to_go,
*             Acc. and X
*
WRITE:
     		jsr	SUEEP        	; Set up the ports for the eeprom.

#IF AUTOERASE
     		jsr  	EWEN         	; Open the eeprom for writing.
#ENDIF

#IF 		9346FORM16
     		INIT16          		; Even the block to be written.
#ENDIF

#IF 		9356FORM16
     		INIT16          		; Even the block to be written.
#ENDIF

#IF 		9366FORM16
     		INIT16          		; Even the block to be written.
#ENDIF

WRLP:
     		lda  	ee_addr   		; eeprom address to be written,
     		inca           		; Update for the
     		sta  	ee_addr   		; next address to be written.
     		deca           		; Restore address to be written
                    			; for this time.

#IFNOT 	AUTOERASE
     		jsr  	ERASE    		; Erase the cell if not autoerase.
#ENDIF	

     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		bset 	DILINE,DIPORT 	; Send 1.
     		jsr  	CLOCK         	; Clock it into the eeprom.
     		bclr 	DILINE,DIPORT 	; Send 0.
     		jsr  	CLOCK         	; Clock it into the eeprom.
     		bset 	DILINE,DIPORT 	; Send 1.
     		jsr  	CLOCK         	; Clock it into the eeprom.

     		jsr  	SENDADR   		; Send eeprom address to eeprom.

     		WRBYTE     			; Send a byte to be written to the 	
					; eeprom.
#IF 		9346FORM16
     		WRBYTE     			; Send a byte to be written to the 	
					; eeprom.
#ENDIF

#IF 		9356FORM16
     		WRBYTE     			; Send a byte to be written to the 	
					; eeprom.
#ENDIF

#IF 		9366FORM16
     		WRBYTE     			; Send a byte to be written to the 	
					; eeprom.
#ENDIF
     		bclr 	DILINE,DIPORT 	; DI low.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom

     		jsr  	WAIT  		; <- Waits here until the byte
                				; is written.
     		tsta       			; Acc still has block_to_go.
     		bne  	WRLP  		; If not done loop again.
WRDONE:
     		jsr  	EWDS         	; Close the eeprom for writing.
     		rts


**********************************************
*  reading - The following are used by the
*            reading routine.
RECEIVE:
     		ldx  	#$8
RCVLP:
     		bset 	CLKLINE,CLKPORT  	; Active clock.
     		bclr 	CLKLINE,CLKPORT  	; Inactive clock.
     		brset DOLINE,DOPORT,RTTY ; Bit from eeprom
                               	; comes in carry.
RTTY:           				; Not really a branch.
     		rola       			; Rotate new bit from carry to
                				; accumulator.
     		decx       			; decrease bit count.
     		bne  	RCVLP 		; If bit count = 0 then acc has 	
					; received byte.
     		coma       			; Complement the whole thing, All
                				; bits come out of the eeprom 	
					; complemented.
     		rts

#MACRO    	READ8
     		jsr  	RECEIVE  		; read 1 byte from eeprom.
     		ldx  	block_to_go 		; Check if finished.
     		beq  	NOSAVE   		; Throw byte away if done.
     		decx          		; If kept decrement the length 	
					; counter.
     		stx  	block_to_go 	; Update the block length counter.
     		ldx  	mem_addr 		; Load the address to store into X.
     		sta  	,x       		; Store read byte to memory.
     		incx          		; Increment the address pointer.
     		stx  	mem_addr 		; Update the address pointer.
NOSAVE:        				; Branch around storage.
#MACROEND

*********************************************;
* READ - This subroutine reads a block of
*        data out of the eeprom and places it
*        in a block of 6805 memory.  It is used
*        with eeproms that do not have the
*        autosequence feature.
*
* INPUTS    - The following memory locations
*             set up as follows.
*         ee_addr   -> contains the eeprom
*                      address where the data
*                      block starts.
*
*         mem_addr  -> contains the absolute
*                      starting
*                      address of the 6805
*                      memory block
*                      destination.
*
*         block_to_go  -> The length of the block,
*                      1 reads one byte,
*                      0 reads none.
* OUTPUTS   - a block of updated memory
* DESTROYS  - ee_addr, mem_addr, block_to_go,
*             Acc. and X
*
*
READ:
     		jsr  	SUEEP        	; Set up the ports for the eeprom.

     		jsr  	EWDS         	; Close the eeprom for writing.
RDNLP:
     		ldx  	block_to_go   	; Read length of block.
     		beq  	RDNDONE       	; If done exit.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		bset 	DILINE,DIPORT 	; Send 11.
     		jsr  	CLOCK2        	; Clock them into the eeprom.
    		bclr 	DILINE,DIPORT 	; Send 0.
     		jsr  	CLOCK         	; Clock it into the eeprom.

    		lda  	ee_addr       	; Bring in eeprom address.
     		inca               	; Update eeprom address
     		sta  	ee_addr       	; for next reading.
     		deca          		; Restore eeprom address for this 	
					; reading.
     		jsr  	SENDADR  		; send eeprom address

     		bclr 	DILINE,DIPORT  	; Bring low for the extra clock
                         		; of read cycle.
		READ8   			; Read a byte out of the eeprom and 
						; into the accumulator.
#IF 		9346FORM16
     		READ8   			; Read byte 2 out of the eeprom and 
						; into the accumulator.
#ENDIF

#IF 		9356FORM16
     		READ8   			; Read byte 2 out of the eeprom and 
						; into the accumulator.
#ENDIF

#IF 		9366FORM16
     		READ8   			; Read byte 2 out of the eeprom and 
						; into the accumulator.
#ENDIF

     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		bra  	RDNLP         	; Go back to see if done reading.
RDNDONE:            			; All done reading.
     		rts

#IF AUTOSEQ
*********************************************;
* RDAUTO - This subroutine reads a block of
*        data out of the eeprom and places it in
*        a block of 6805 memory.  It functions
*        faster than the READ routine above and
*        can only be used with the newer eeproms
*        that automatically cycle to the next
*        register, the "autosequence" or 
*        "autoincrement" feature.
*
* INPUTS    - The following memory locations
*             set up as follows.
*         ee_addr   -> contains the eeprom
*                   address where the data
*                   block starts.
*
*         mem_addr  -> contains the absolute
*                   starting address of the 6805
*                   memory block destination.
*
*         block_to_go  -> The length of the block,
*                   1 writes one byte, 0 writes
*                   none.
* OUTPUTS   - a block of updated memory
* DESTROYS  - ee_addr, mem_addr, block_to_go,
*             Acc. and X.
*
* USES - "RECEIVE" which is defined above.
*
RDAUTO:
     		jsr  	SUEEP        	; Set up the ports for the eeprom.

     		jsr  	EWDS         	; Close the eeprom for writing.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		bset 	DILINE,DIPORT 	; Send 11.
     		jsr  	CLOCK2        	; Clock them into the eeprom.
     		bclr 	DILINE,DIPORT 	; Send 0.
     		jsr  	CLOCK         	; Clock it into the eeprom.

     		lda  	ee_addr   		; Bring in eeprom address.
     		jsr  	SENDADR   		; Send it out.

     		bclr 	DILINE,DIPORT  	; Bring low for the extra clock
                         		; of read cycle.
RDALP:
     		ldx  	block_to_go  	; Bring in length left to send.
     		tstx           		; Check for done.
     		beq  	RDADONE   		; If done exit.
     		decx           		; Decrement length left.
     		stx  	block_to_go  	; Update length left.
     		jsr  	RECEIVE   		; Receive a byte from the eeprom.
     		ldx  	mem_addr  		; Load in the place in memory to
                    			; put the byte.
     		sta  	,x  			; <- Change store command here if 	
					; memory is above $100.
     		incx            		; increment the memory pointer.
     		stx  	mem_addr   		; Update the memory pointer.
     		bra  	RDALP      		; Loop back until done.
RDADONE:
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		rts
#ENDIF

*********************************************;
* START - Sample calling of routines.

BSTART  	EQU  	0    			; Start eeprom addresses for these
						; examples.
BL_LEN  	EQU  	$80  			; Length of block for these 		
					; examples.
STARTRD:
    		lda   #BSTART    		; Start reading eeprom at
                     			; address BSTART.
    		sta   ee_addr    		; Place first eeprom address in 	
					; ee_addr.
    		lda   #data      		; Load in start address of receiving
                     			; memory.
    		sta   mem_addr   		; Place start address in mem_addr.
    		lda   #BL_LEN    		; Length of block to read.
    		sta   block_to_go		; Store block length.

#IF AUTOSEQ
    		jsr  	RDAUTO     		; Read the eeprom using 		
					; autosequencing.
#ELSEIF
    		jsr  	READ     		; Read the W/O Autosequencing 	
					; eeprom.
#ENDIF
    		bra  	$        		; jump to this location
                  			; (do nothing else).

STARTWR:
    		lda   #BSTART    		; Start writing eeprom at
                     			; address BSTART.
    		sta   ee_addr    		; Place first eeprom address in 	
					; ee_addr.
    		lda   #data      		; Load in start address of block in
                     			; memory.
    		sta   mem_addr   		; Place start address in mem_addr.
    		lda   #BL_LEN    		; Length of block to write.
    		sta   block_to_go		; Store block length.
     		jsr  	WRITE  		; WRITE the block.
     		bra  	$     		; (do nothing else).

STARTERAL:
    		jsr  	ERAL 			; Erases the entire serial eeprom
LOOP3:
     		bra  	$     		; (do nothing else).

STARTWRL:
    		lda  	#$a5  		; (write $a5 to form 8 eeprom.)

#IF 		9346FORM16
    		ldx  	#$c3  		; write $a5c3 to form 16 eeprom.
#ENDIF

#IF 		9356FORM16
    		ldx  	#$c3  		; write $a5c3 to form 16 eeprom.
#ENDIF

#IF 		9366FORM16
    		ldx  	#$c3  		; write $a5c3 to form 16 eeprom.
#ENDIF
    		jsr  	WRAL 			; 0xa5 to all memory locations in
              				; the eeprom.
*     	bra  	$     		; (do nothing else).

STARTERSE:
    		lda  	#$05  		; Bring in 5 as location to be 	
					; erased.
    		jsr  	ERASE 		; Erases memory location 5 of the
               				; eeprom.
     		bra  	$     		; (do nothing else).

	ORG VECTORS

VECSPI:  	fdb   STARTRD   		; SPI VECTOR
VECSCI:  	fdb   STARTRD   		; SCI VECTOR
VECTMR:  	fdb   STARTRD   		; TIMER VECTOR
VECIRQ:  	fdb   STARTRD   		; IRQ VECTOR
VECSWI:  	fdb   STARTRD   		; SWI VECTOR
VECRST:  	fdb   STARTRD   		; START VECTOR

	


