* APPENDIX B - SPI Polling to eeprom application source



RAM  		EQU  	$50       	; RAM starts at $50
ROM  		EQU  	$100      	; ROM Starts at $100
VECTORS  	EQU  	$1ff4 	; Reset and interrupt vectors start 	
				; at $1ff4

******************************************
* Eeprom type and configuration switches
*
#SETNOT 	9346FORM8   	; 9346 eeprom, 1 byte format.
#SETNOT 	9346FORM16  	; 9346, 2 byte word format.
#SETNOT 	9356FORM16  	; 9356, 2 byte word format.
#SET 		9366FORM16  	; 9366, 2 byte word format.

#SET 		AUTOERASE  		; For eeproms that do not need to
                			; erase before writing.
#SET 		AUTOSEQ    		; For eeproms that automatically
                			; sequence to the next cell when
                			; being read.

******************************************
*
* RAM - variables
*
*
*

    	ORG  RAM

ee_addr   	ds  	1 		; eeprom address stored here.
mem_addr    ds  	1 		; Block index stored here.
block_to_go ds  	1 		; Block length stored here.

data  	ds  	$ad    	; Rest of data space is data to be stored.

*
* PROGRAM
*
* The main subroutines are READ, EWEN, EWDS,
* WRITE, WRAL, ERASE, and ERAL.  SLACK, WAIT,
* CLOCK, and SHUFFLE support
* these.
*
* Port locations follow.
*
*
CSPORT   	EQU  	0 		; Eeprom Chip Select.
CSLINE   	EQU  	5 		; porta.5, an output line.

SPCR     	EQU  	$0a 		; Location of SPI control reg.
SPSR     	EQU  	$0b 		; Location of SPI status reg.
SPIDAT   	EQU  	$0c 		; Location of SPI data reg.


    	ORG  ROM

******************************************
* Command set

#if 		9346FORM8      	; Command set for 9346 in the byte wide 	
				; form.
MASK      	equ  	%01111111  	; Mask of valid address bits
READ1     	equ  	%00000110  	; READ command padded to 16 bits.
READ2     	equ  	%00000000
EWEN1     	equ  	%00000010  	; Write enable command padded to 16 bits.
EWEN2     	equ  	%01100000
EWDS1     	equ  	%00000010  	; Write protect command padded to 16 bits.
EWDS2     	equ  	%00000000
WRITE1    	equ  	%00000010  	; Write command padded to 16 bits.
WRITE2    	equ  	%10000000
WRAL1     	equ  	%00000010  	; Write all command padded to 16 bits.
WRAL2     	equ  	%00100000
ERASE1    	equ  	%00000011  	; Erase cell command padded to 16 bits.
ERASE2    	equ  	%10000000
ERAL1     	equ  	%00000010  	; Erase all command padded to 16 bits.
ERAL2     	equ  	%01000000
#endif

#if 		9346FORM16     	; Command set for 9346 in the 16 bit wide 
					; form.
MASK      	equ  	%00111111  	; Mask of valid address bits
READ1     	equ  	%00000011  	; READ command padded to 16 bits.
READ2     	equ  	%00000000
EWEN1     	equ  	%00000001  	; Write enable command padded to 16 bits.
EWEN2     	equ  	%00110000
EWDS1     	equ  	%00000001  	; Write protect command padded to 16 bits.
EWDS2     	equ  	%00000000
WRITE1    	equ  	%00000001  	; Write command padded to 16 bits.
WRITE2    	equ  	%01000000
WRAL1     	equ  	%00000001  	; Write all command padded to 16 bits.
WRAL2     	equ  	%00010000
ERASE1    	equ  	%00000001  	; Erase cell command padded to 16 bits.
ERASE2    	equ  	%11000000
ERAL1     	equ  	%00000001  	; Erase all command padded to 16 bits.
ERAL2     	equ  	%00100000
#endif

#if 		9356FORM16     	; Command set for 9356 in the 16 bit wide 
					; form.
MASK      	equ  	%01111111  	; Mask of valid address bits
READ1     	equ  	%00001100  	; READ command padded to 16 bits.
READ2     	equ  	%00000000
EWEN1     	equ  	%00000100  	; Write enable command padded to 16 bits.
EWEN2     	equ  	%11000000
EWDS1     	equ  	%00000100  	; Write protect command padded to 16 bits.
EWDS2     	equ  	%00000000
WRITE1    	equ  	%00000101  	; Write command padded to 16 bits.
WRITE2    	equ  	%00000000
WRAL1     	equ  	%00000100  	; Write all command padded to 16 bits.
WRAL2     	equ  	%01000000
ERASE1    	equ  	%00000111  	; Erase cell command padded to 16 bits.
ERASE2    	equ  	%00000000
ERAL1     	equ  	%00000100  	; Erase all command padded to 16 bits.
ERAL2     	equ  	%10000000
#endif

#if 		9366FORM16     	; Command set for 9366 in the 16 bit wide 
					; form.
MASK      	equ  	%11111111  	; Mask of valid address bits
READ1     	equ  	%00001100  	; READ command padded to 16 bits.
READ2     	equ  	%00000000
EWEN1     	equ  	%00000100  	; Write enable command padded to 16 bits.
EWEN2     	equ  	%11000000
EWDS1     	equ  	%00000100  	; Write protect command padded to 16 bits.
EWDS2     	equ  	%00000000
WRITE1    	equ  	%00000101  	; Write command padded to 16 bits.
WRITE2    	equ  	%00000000
WRAL1     	equ  	%00000100  	; Write all command padded to 16 bits.
WRAL2     	equ  	%01000000
ERASE1    	equ  	%00000111 	; Erase cell command padded to 16 bits.
ERASE2    	equ  	%00000000
ERAL1     	equ  	%00000100  	; Erase all command padded to 16 bits.
ERAL2     	equ  	%10000000
#endif


*********************************************;
* WAIT - This routine delays the next command
*        to the eeprom until the most recent
*        write or erase has finished.
*        If in a write or erase
*        cycle the routine loops.  One
*        write or erase takes 4
*        milliseconds.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - nothing
*
WAIT:
     		bset 	CSLINE,CSPORT  	; Select.

     		clr  	SPIDAT  		; Send 8 more don't care zeros.
     		brclr 7,SPSR,$  		; Loop here until eeprom ready.
     		tst  	SPIDAT   		; Is eeprom ready?  Zero if not.
     		beq  	WAIT   		; if not ready send more clocks and 
						; zeros.
     		bclr 	CSLINE,CSPORT  	; De-select eeprom is ready.
     		rts

*********************************************;
* IDIO - This routine handles the idiosyncratic
*        requirements of the particular test
*        hardware used.  It may be deleted
*        in most applications
IDIO:
     		bset 	4,CSPORT+4  	; Output port 0.4
     		bset 	4,CSPORT 		; Pulls up the "SS" line.
     		bset 	7,CSPORT+4  	; Output port 0.7
     		bset 	7,CSPORT 		; Pulls up the "RESET" line.
     		rts

*********************************************;
* SUSPI - Sets up the eeprom IO port and the
*         SPI to communicate with the eeprom
*         by polling.
*         Other tasks can share the SPI.
*
* INPUTS   - none
* OUTPUTS  - DDRA,SPI
* DESTROYS - Accumulator.
*
SUSPI:
     		bset 	CSLINE,CSPORT+4 	; Chip select line is output.
     		bclr 	CSLINE,CSPORT   	; Chip select is low de-selected.
     		lda  	#%01010000 		; SPI enabled phase 0.
     		sta  	SPCR       		; SPI control register, SPI set up.
     		rts

*********************************************;
* EESEND - sends a byte to the eeprom through
*           the SPI.
*
* INPUTS    - accumulator, send to SPI
* OUTPUTS   - accumulator, response from SPI
*
* DESTROYS  - Accumulator
*
EESEND:
        	sta 	SPIDAT   		; Accumulator goes out the SPI.
        	brclr	7,SPSR,$  		; Should loop 3 times.
        	lda	SPIDAT   		; What comes out of the SPI is
                         		; placed in the accumulator.
        	rts

*********************************************;
* EWEN - This subroutine enables erase and write
*        operations.  It in effect unlocks the
*        eeprom so that its cells may be
*        changed.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - nothing
*
EWEN:
     		jsr  	SUSPI     		; Ensure that the SPI is set up.
     		lda  	#EWEN1    		; Load first part of EWEN command.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		jsr  	EESEND    		; Send the command out the SPI.
     		lda  	#EWEN2    		; Load the second part of EWEN 	
					; command.
     		jsr  	EESEND    		; Send the command out the SPI.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		rts
 
*********************************************;
* EWDS - This subroutine disables erase and
*        write operations so that data cannot be
*        inadvertently corrupted.  It in effect
*        locks the eeprom so that its cells
*        cannot be changed.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - nothing
*
EWDS:
     		jsr  	SUSPI     		; Ensure that the SPI is set up.
     		lda  	#EWDS1    		; Load first part of the EWDS 	
					; command.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		jsr  	EESEND    		; Send EWDS1 out the SPI.
     		lda  	#EWDS2    		; Load second part of the EWDS 	
					; command.
     		jsr  	EESEND    		; Send EWDS2 out the SPI.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		rts

*********************************************;
* ERAL - This subroutine erases the entire
*        eeprom.  An erased 93x6 cell will
*        put a high level on the DO line when
*        read, but due to inverting in READ,
*        the result will arrive as 0x00 in
*        6805 memory.  ERAL calls EWEN to
*        allow erasure.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - all contents of eeprom
*
ERAL:
     		jsr 	SUSPI     		; Ensure that the SPI is set up.
     		jsr  	EWEN      		; "OPEN" the eeprom for writing and 
						; erasure.
     		lda  	#ERAL1    		; Load the first part of the ERAL 	
					; command.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		jsr  	EESEND    		; Send ERAL1 out the SPI.
     		lda  	#ERAL2    		; Load the second part of the ERAL 	
					; command.
     		jsr  	EESEND    		; Send ERAL2 out the SPI.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		jsr  	WAIT      		; Wait until eeprom is ready.
     		rts

*********************************************;
* WRAL - In FORM8 eeproms this subroutine
*        writes the byte in the accumulator to
*        every byte of the Eeprom.  In FORM16
*        eeproms the accumulator is written to
*        the most significant byte the X
*        register is written to the less
*        significant byte.
*
* INPUTS    - Accumulator ( and X for FORM16)
* OUTPUTS   - none
* DESTROYS  - accumulator, ee_addr, and mem_addr .
*

#MACRO 	WRAL16
     		lda  	mem_addr 		; Load Second byte of word to be 	
					; written.
     		jsr  	EESEND   		; Send that second byte out the SPI 
						; to eeprom.
#MACROEND

WRAL:
     		sta  	ee_addr   		; Store low order byte in ee_addr.
     		stx  	mem_addr  		; Store high order byte in mem_addr.
     		jsr  	SUSPI     		; Ensure that the SPI is set up.

#IFNOT 	AUTOERASE
     		jsr  	ERAL  		; if not autoerase, erase the eeprom 
						; first.
#ELSEIF
     		jsr  	EWEN  		; if an autoerase eeprom open it for 
						; writing.
#ENDIF
     		lda  	#WRAL1		; Load Write all #1 command.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		jsr  	EESEND		; Send Write All command #1 out the 
						; SPI.
     		lda  	#WRAL2		; Load Write All #2 command.
     		jsr  	EESEND		; Send Write all command #2 out the 
						; SPI.
     		lda  	ee_addr		; Load the low byte to be written.
     		jsr  	EESEND 		; Send it out the SPI.

#IF 		9346FORM16
     		WRAL16     			; Send out the high byte if a form 	
					; 16 eeprom.
#ENDIF

#IF 		9356FORM16
     		WRAL16     			; Send out the high byte if a form 	
					; 16 eeprom.
#ENDIF

#IF 		9366FORM16
     		WRAL16     			; Send out the high byte if a form 	
					; 16 eeprom.
#ENDIF
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom

     		jsr  	WAIT  		; <- Waits here for erasure to
                				; finish.

     		jsr  	EWDS  		; "Close" or write protect the 	
					; eeprom.
     		rts

*********************************************;
* ERASE - This subroutine Erases an eight
*         cell byte or 16 cell word in the
*         Eeprom.  The address of the cell is
*         located in the accumulator.  The
*         accumulator is returned unchanged.
*
* INPUTS    - Eeprom address for erasure in Acc.
* OUTPUTS   - none
* DESTROYS  - X
*

ERASE:
     		jsr  	SUSPI     		; Ensure that the SPI is set up.
     		jsr  	EWEN      		; Open the eeprom for writing.
     		lda  	#ERASE1   		; Load with Erase #1 command.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		jsr  	EESEND    		; Send Erase #1 command out SPI.
     		txa            		; Copy address to X for storage.
     		and  	#MASK     		; AND address with mask.
     		ora  	#ERASE2   		; OR address with ERASE #2 command.
     		jsr  	EESEND    		; Send out SPI.

     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom

     		jsr  	WAIT  		; <- Waits here for erasure to
                				; finish.
     		txa        			; Return eeprom address to 		
					; accumulator.
     		rts

**********************************************
* Write macros ------------------------------

#MACRO	WRBYTE
     		ldx  	mem_addr  		; Load pointer reg with address
                    			; of byte to be sent next.
     		lda  	,x        		; Bring that byte into accumulator.
     		incx           		; Increment pointer for next byte.
     		stx  	mem_addr  		; Update with address of next byte 	
					; to be sent.
     		jsr  	EESEND    		; Send byte out SPI.
     		lda  	block_to_go  	; Load the length left to be sent.
     		deca         		; Dec length and check if done.
     		sta  	block_to_go  	; Update the length of block to be 	
					; sent.
#MACROEND

#MACRO 	INIT16
        	lda	block_to_go  	; Load the length left to be sent.
        	beq   WRDONE    		; If Zero finish.
        	rora            		; Place least significant bit in 	
					; carry.
        	bcc   LEN_OK  		; Ensure that block_to_go
        	rola            		; starts as an even number.
        	inca            		; If not increment to an even 	
					; number.
        	sta   block_to_go 	; Update to the new even number.
LEN_OK:
#MACROEND

*********************************************;
* WRITE - This subroutine Writes a block of
*         eight cell bytes to the Eeprom.
*
* INPUTS    - The following memory locations
*             set up as follows.
*         ee_addr  -> contains the absolute
*                      address of where the
*                      data will start in
*                      the eeprom.
*         mem_addr  -> contains the absolute
*                      starting address of the
*                      block of memory
*                      to be written to eeprom.
*         block_to_go -> The length of the block,
*                      1 writes one byte,
*                      0 writes none.
* OUTPUTS   - none
* DESTROYS  - ee_addr, mem_addr , block_to_go,
*             Acc. and X
*
WRITE:
     		jsr  	SUSPI     		; Ensure that the SPI is set up.

#IF 		AUTOERASE
     		jsr  EWEN      		; Open the eeprom for writing.
#ENDIF

#IF 		9346FORM16
     		INIT16     			; Adjust the length of the block to 
						; be sent.
#ENDIF

#IF 		9356FORM16
     		INIT16     			; Adjust the length of the block to 
						; be sent.
#ENDIF

#IF 		9366FORM16
     		INIT16     			; Adjust the length of the block to 
						; be sent.
#ENDIF

WRLP:     					; <- This is where the loop starts 	
					; for repetitive writes to the 		
				; eeprom until the block to be
          					; written is zero.
#IFNOT 	AUTOERASE
     		lda  	ee_addr  		; eeprom address to be written,
     		jsr  	ERASE    		; Erase the cell if not autoerase.
#ENDIF

     		lda  	#WRITE1  		; Load the first part of the write 	
					; command.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		jsr  	EESEND   		; Send the first part of the write 	
					; command out SPI.
     		lda  	ee_addr  		; eeprom address to be written.
     		inca          		; Increment it for next byte.
     		sta  	ee_addr  		; Update eeprom address to be 	
					; written.
     		deca          		; Decrement eeprom address to be 	
					; written for this byte.
     		and  	#MASK    		; AND with address mask for this 	
					; type and form of eeprom.
     		ora  	#WRITE2  		; OR with Write command #2.
     		jsr  	EESEND   		; Send address and WRITE2 out SPI.

     		WRBYTE        		; Send a byte to be written out SPI.

#IF 		9346FORM16
     		WRBYTE        		; If 16 bit form Send the second 	
					; byte to be written out SPI.
#ENDIF

#IF 		9356FORM16
     		WRBYTE        		; If 16 bit form Send the second 	
					; byte to be written out SPI.
#ENDIF

#IF 		9366FORM16
     		WRBYTE        		; If 16 bit form Send the second 	
					; byte to be written out SPI.
#ENDIF
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom

     		jsr  	WAIT  		; <- Waits here until the byte
                				; is written.
     		tsta       			; Acc still has block_to_go.
     		bne  	WRLP  		; Loop until the block left is zero.
WRDONE:         				; All done writing.
     		jsr  	EWDS  		; "Close" or Write protect the 	
					; eeprom.
     		rts

**********************************************
*  reading - The following is used to read
*            form 16 configured eeproms.
*
#MACRO 	RD_BYTE
     		jsr  	EESEND      	; Read a byte from the eeprom 	
					; through the SPI.
     		ldx  	block_to_go 	; Load the length left to read.
     		beq  	NOSAVE      	; Only store if length is left.
     		decx             		; Decrement the block counter.
     		stx  	block_to_go 	; Update the block to go length.
     		ldx  	mem_addr    	; Load location where the byte from 
						; eeprom is to go.
     		sta  	,x          	; Store the byte from eeprom to 	
					; memory.
     		incx             		; Increment the location for the 	
					; next byte.
     		stx  	mem_addr    	; Update the memory address for the 
						; next read.
NOSAVE:               			; jump to here if end of block 	
					; occurs.
#MACROEND

*********************************************;
* READ - This subroutine reads a block of
*        data out of the eeprom and places it
*        in a block of 6805 memory.  It has the
*        autosequence feature as an option.
*
* INPUTS    - The following memory locations
*             set up as follows.
*         ee_addr  -> contains the eeprom
*                      address where the data
*                      block starts.
*
*         mem_addr  -> contains the absolute
*                      starting
*                      address of the 6805
*                      memory block
*                      destination.
*
*         block_to_go -> The length of the block,
*                      1 reads one byte,
*                      0 reads none.
* OUTPUTS   - a block of updated memory
* DESTROYS  - ee_addr, mem_addr, block_to_go,
*             Acc. and X
*
*
READ:
     		jsr  	SUSPI     		; Ensure that the SPI is set up.

     		jsr  	EWDS      		; Ensure that the eeprom is write 	
					; protected.
RDNLP:
     		ldx  	block_to_go 	; Load in the length of block to 	
					; read.
     		tstx             		; Test for a zero length block.
     		beq  	RDNDONE     	; Test length to see if done.
     		decx             		; Decrement the length of the block.
     		stx  	block_to_go 	; Update block length for next loop.

#if 		9366FORM16
     		lda  	ee_addr   		; Bring in eeprom address
     		lsla           		; Place MS Bit in carry.
     		clra           		; Zero out the accumulator.
     		rola           		; MS Bit of ee_address is LS Bit of 
						; accumulator.
     		ora  	#READ1    		; Overlay first part of read 		
					; command.
#ELSEIF
     		lda  	#READ1    		; Load the first part of read 	
					; command.
#ENDIF
     		tst  	SPSR        	; clean out the SPI receiver.
     		tst  	SPIDAT      	; Ensure SPI has no old data in it.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		jsr  	EESEND      	; Send READ1 command.
     		lda  	ee_addr     	; Load in eeprom address.
     		and  	#MASK       	; Mask in only valid address bits.
     		lsla             		; Shift left to create dummy clock
                 	     			; idiosyncratic to READ.
     		ora  	#READ2      	; OR address with READ#2.
     		jsr  	EESEND      	; Send READ2 out the SPI.
     		lda  	ee_addr     	; Load accumulator with eeprom 	
					; address.
     		inca             		; Increment eeprom address for next 
						; pass.
     		sta  	ee_addr     	; Update eeprom address.
     		clra             		; Clear the accumulator to read 	
					; eeprom.
     		jsr  	EESEND      	; Read first byte.
     		ldx  	mem_addr    	; Load X with the location to store 
						; read byte.
     		sta  	,x          	; Store the read byte.
     		incx             		; Increment X in preparation for 	
					; next read.
     		stx  	mem_addr    	; Update the memory address.

#IF 		AUTOSEQ
WRLOOP:            			; Tighter loop for autosequence 	
					; eeproms.
     		RD_BYTE       		; Read a byte from the eeprom + 	
					; Store it.
     		bne  	WRLOOP   		; If block length = 0, all done, 	
					; else loop.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		bra  	RDNDONE  		; Branch to out.
#ENDIF

#IF 		9346FORM16
     		RD_BYTE       		; Read a byte from the eeprom + 	
					; Store it.
#ENDIF

#IF 		9356FORM16
     		RD_BYTE       		; Read a byte from the eeprom + 	
					; Store it.
#ENDIF

#IF 		9366FORM16
     		RD_BYTE       		; Read a byte from the eeprom + 	
					; Store it.
#ENDIF
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		bra  	RDNLP    		; Branch to set up command and 	
					; address necessary
                   			; for non autosequenced eeproms.
RDNDONE:    				; Branch to here when all done.
     		rts


*********************************************;
* START - Sample calling of routines.
*
BSTART  	EQU  	0    			; Start eeprom addresses for these 	
					; examples.
BL_LEN  	EQU  	$80  			; Length of block for these 		
					; examples.

STARTRD:
    		jsr  	IDIO        	; ensure the ports are set up
                     			; for this particular test set up.
    		lda   #BSTART    		; Start reading eeprom at
                     			; address BSTART.
    		sta   ee_addr    		; Place first eeprom address in 	
					; ee_addr.
    		lda   #data      		; Load in start address of receiving
                     			; memory.
    		sta   mem_addr   		; Place start address in mem_addr.
    		lda   #BL_LEN    		; Length of block to read in.
    		sta   block_to_go		; Store block length.
    		jsr  	READ     		; Read the eeprom.
    		bra  	$        		; jump to this location
                  			; (do nothing else).

STARTWR:
     		jsr  	IDIO       		; ensure the ports are set up
                     			; for this particular test set up.
     		lda  	#data      		; Load in start address of receiving
                     			; memory.
    		sta  	mem_addr   		; Place start address in mem_addr.
     		lda  	#BSTART    		; Start writing eeprom with bytes at
                     			; address BSTART and up.
    		sta  	ee_addr    		; Place first eeprom address in 	
					; ee_addr.
     		lda  	#BL_LEN    		; Length of block to write to 	
					; eeprom.
     		sta  	block_to_go		; Store block length.
     		jsr  	WRITE      		; Write the block to the eeprom.
     		bra  	$          		; jump to this location
                     			; (do nothing else).

STARTERAL:
    		jsr   IDIO       		; ensure the ports are set up
                     			; for this particular test set up.
    		jsr  	ERAL     		; Erases the entire serial eeprom
    		bra  	$        		; jump to this location
                  			; (do nothing else).

STARTWRL:
    		jsr  	IDIO       		; ensure the ports are set up
                    			; for this particular test set up.
    		lda  	#$a5  		; (write $a5 to form 8 eeprom.)

#IF 		9346FORM16
    		ldx  	#$c3  		; write $a5c3 to form 16 eeprom.
#ENDIF

#IF 		9356FORM16
    		ldx  	#$c3  		; write $a5c3 to form 16 eeprom.
#ENDIF

#IF 		9366FORM16
    		ldx  	#$c3  		; write $a5c3 to form 16 eeprom.
#ENDIF
    		jsr  	WRAL   		; 0xa5 to all memory locations in
                				; the eeprom.
*    		bra  	$     		; jump to this location
                				; (do nothing else).


STARTERSE:
    		lda  	#$05  		; Load A with the eeprom address to 
						; be erased.
    		jsr  	ERASE 		; Erases memory location 5 of the
               				; eeprom.
    		bra  	$        		; jump to this location
                  			; (do nothing else).


	ORG VECTORS
	
VECSPI:  	fdb   STARTRD   		; SPI VECTOR
VECSCI:  	fdb   STARTRD   		; SCI VECTOR
VECTMR:  	fdb   STARTRD   		; TIMER VECTOR
VECIRQ:  	fdb   STARTRD   		; IRQ VECTOR
VECSWI:  	fdb   STARTRD   		; SWI VECTOR
VECRST:  	fdb   STARTRD   		; START VECTOR

	


