* APPENDIX C - SPI to eeprom using interrupt


*~~~~~~~~~~~~~~~MACRO~~~FOR~~WRITING~~~~~~~~~~~~~~~~~~~~~~~~~
* This writes a block of memory starting at absolute address
* "RAM_start" of length "length" to the eeprom starting at
* its absolute address "ee_start."

#MACRO 	WRBLOCK 	ee_start,RAM_start,length
     		jsr  	CK_CLR     		; Ensure the eeprom is free.
     		lda  	#%2        		; Get Start of block in memory.
     		sta  	mem_addr   		; Place memory start in proper 	
					; place.
     		lda  	#%1        		; Get Start of block in destination 
						; eeprom.
     		sta  	ee_addr    		; Place eeprom destination start in 
						; proper place.
     		lda  	#%3           	; Get the full block length.
     		sta  	block_to_go   	; Place Block length in proper 	
					; place.
     		jsr  	WRITE         	; Write block from memory to eeprom.
#MACROEND

RAM  		EQU  	$50       		; RAM starts at $50
ROM  		EQU  	$100      		; ROM starts at $100
VECTORS  	EQU  	$1ff4 		; RESET and interrupt vectors start 
						; at $1ff4.

******************************************
* Eeprom type and configuration switches
*
#SETNOT 	9346FORM8   		; 9346 eeprom, 1 byte format.
#SETNOT 	9346FORM16  		; 9346, 2 byte word format.
#SETNOT 	9356FORM16  		; 9356, 2 byte word format.
#SET 		9366FORM16  		; 9366, 2 byte word format.
*
* Use with AUTOERASE eeproms only.
*
#SETNOT 	AUTOSEQ    			; For eeproms that automatically
                				; sequence to the next cell when
                				; being read.


******************************************
*
* RAM - variables
*
*
*
	ORG  	RAM
ee_addr   	ds  	1 			; eeprom address stored here.
mem_addr    ds  	1 			; Block index stored here.
block_to_go ds  	1 			; Block length stored here.
flag        ds  	1 			; Flags for eeprom status.

* Two flags are usually used
WR  		equ  	0       		; Set for Write, reset for erase.
m_to_pr  	equ  	1  			; More to program flag.
data  	ds  	$ad    		; Rest of data space is data to be 	
					; stored.

 	ORG  	ROM

******************************************
* Command set
#if 		9346FORM8      		; Command set for 9346 in the byte 	
					; wide form.
MASK      	equ  	%01111111  		; Mask of valid address bits
READ1     	equ  	%00000110  		; READ command padded to 16 bits.
READ2     	equ  	%00000000
EWEN1     	equ  	%00000010  		; Write enable command padded to 16 
						; bits.
EWEN2     	equ  	%01100000
EWDS1     	equ  	%00000010  		; Write protect command padded to 16 
						; bits.
EWDS2     	equ  	%00000000
WRITE1    	equ  	%00000010  		; Write command padded to 16 bits.
WRITE2    	equ  	%10000000
WRAL1     	equ  	%00000010  		; Write all command padded to 16 	
					; bits.
WRAL2     	equ  	%00100000
ERASE1    	equ  	%00000011  		; Erase cell command padded to 16 	
					; bits.
ERASE2    	equ  	%10000000
ERAL1     	equ  	%00000010  		; Erase all command padded to 16 	
					; bits.
ERAL2     	equ  	%01000000
#endif

#if 		9346FORM16     		; Command set for 9346 in the 16 bit 
						; wide form.
MASK      	equ  	%00111111  		; Mask of valid address bits
READ1     	equ  	%00000011  		; READ command padded to 16 bits.
READ2     	equ  	%00000000
EWEN1     	equ  	%00000001  		; Write enable command padded to 16 
						; bits.
EWEN2     	equ  	%00110000
EWDS1     	equ  	%00000001  		; Write protect command padded to 16 
						; bits.
EWDS2     	equ  	%00000000
WRITE1    	equ  	%00000001  		; Write command padded to 16 bits.
WRITE2    	equ  	%01000000
WRAL1     	equ  	%00000001  		; Write all command padded to 16 	
					; bits.
WRAL2     	equ  	%00010000
ERASE1    	equ  	%00000001  		; Erase cell command padded to 16 	
					; bits.
ERASE2    	equ  	%11000000
ERAL1     	equ  	%00000001  		; Erase all command padded to 16 	
					; bits.
ERAL2     	equ  	%00100000
#endif

#if 		9356FORM16     		; Command set for 9356 in the 16 bit 
						; wide form.
MASK      	equ  	%01111111  		; Mask of valid address bits
READ1     	equ  	%00001100  		; READ command padded to 16 bits.
READ2     	equ  	%00000000
EWEN1     	equ  	%00000100  		; Write enable command padded to 16 
						; bits.
EWEN2     	equ  	%11000000
EWDS1     	equ  	%00000100  		; Write protect command padded to 16 
						; bits.
EWDS2     	equ  	%00000000
WRITE1    	equ  	%00000101  		; Write command padded to 16 bits.
WRITE2    	equ  	%00000000
WRAL1     	equ  	%00000100  		; Write all command padded to 16 	
					; bits.
WRAL2     	equ  	%01000000
ERASE1    	equ  	%00000111  		; Erase cell command padded to 16 	
					; bits.
ERASE2    	equ  	%00000000
ERAL1     	equ  	%00000100  		; Erase all command padded to 16 	
					; bits.
ERAL2     	equ  	%10000000
#endif

#if 		9366FORM16     		; Command set for 9366 in the 16 bit 
						; wide form.
MASK      	equ  	%11111111  		; Mask of valid address bits
READ1     	equ  	%00001100  		; READ command padded to 16 bits.
READ2     	equ  	%00000000
EWEN1    	equ  	%00000100  		; Write enable command padded to 16 
						; bits.
EWEN2     	equ  	%11000000
EWDS1     	equ  	%00000100  		; Write protect command padded to 16 
						; bits.
EWDS2     	equ  	%00000000
WRITE1    	equ  	%00000101  		; Write command padded to 16 bits.
WRITE2    	equ  	%00000000
WRAL1     	equ  	%00000100  		; Write all command padded to 16 	
					; bits.
WRAL2     	equ  	%01000000
ERASE1    	equ  	%00000111  		; Erase cell command padded to 16 	
					; bits.
ERASE2    	equ  	%00000000
ERAL1     	equ  	%00000100  		; Erase all command padded to 16 	
					; bits.
ERAL2     	equ  	%10000000
#endif

CMDLST     					; Command list.
     		DB   	WRAL1     		; WRAL Write All is #0 in the 	
					; command list.
     		DB   	WRAL2
     		DB   	ERAL1     		; ERAL Erase All is #1 in the 	
					; command list.
     		DB   	ERAL2
     		DB   	ERASE1    		; ERASE one cell is #2 in the 	
					; command list.
     		DB   	ERASE2
     		DB   	WRITE1    		; WRITE a block of cells is #3 in
     		DB   	WRITE2    		; the command list.
     		DB   	EWDS1     		; EWDS Write protect or close is #4 
     		DB   	EWDS2     		; in the command list.
    		DB   	EWEN1     		; EWEN Enable write or open is #5 in 
     		DB   	EWEN2     		; the command list.

*
* PROGRAM
*
* The main subroutines are READ, EWEN, EWDS,
* WRITE, WRAL, ERASE, and ERAL.
*

CSPORT   	EQU  	0 			; Eeprom Chip Select.
CSLINE   	EQU  	5 			; porta.5, an output line.

SPCR     	EQU  	$0a 			; Location of SPI control reg.
SPSR     	EQU  	$0b 			; Location of SPI status reg.
SPIDAT   	EQU  	$0c 			; Location of SPI data reg.

SPIIRON  	EQU  	%11010011 		; SPI and interrupt on
                        		; with lowest possible
                        		; baud rate.

*********************************************;
* SETUP - This routine initializes the flags
*        to the preset inactive condition.
*
SETUP:
     		bclr 	m_to_pr,flag  	; Initialize to no more to
                        		; program into the eeprom.
     		bclr 	WR,flag       	; Not writing at initialization.
     		rts

*********************************************;
* IDIO - This routine handles the idiosyncratic
*        requirements of the particular test
*        hardware used.  It may be deleted
*        in most applications
IDIO:
     		bset 	4,CSPORT+4  	; Output in this application.
     		bset 	4,CSPORT    	; Pulls up the "SS" line.
     		bset 	7,CSPORT+4  	; Output in this application.
     		bset 	7,CSPORT    	; Pulls up the "RESET" line.
     		rts

*********************************************;
* SUSPI - Sets up the eeprom IO port and the
*         SPI to communicate with the eeprom
*         by polling.
*         Other tasks can share the SPI.
*
* INPUTS   - none
* OUTPUTS  - DDRA,SPI
* DESTROYS - Accumulator.
*
SUSPI:
     		bset 	CSLINE,CSPORT+4  	; Output for Chip Select.
     		bclr 	CSLINE,CSPORT    	; Initialize to not selected.
     		lda  	#%01010000       	; SPI enabled phase 0.
     		sta  	SPCR             	; Set up the SPI to phase 0.
     		rts

*********************************************;
* SUSPIR - Sets up the SPI to communicate
*          with the eeprom with interrupts.
*          This is used to determine when the
*          eeprom is ready.
*
* INPUTS   - none
* OUTPUTS  - DDRA,SPI
* DESTROYS - Accumulator.
*
SUSPIR:
     		lda  	#SPIIRON 		; SPI enabled phase 0.
     		sta  	SPCR     		; Set up SPI with interrupt.
     		cli           		; Enable the interrupt.
     		rts

*********************************************
* SUALT - This is an example alternate
*         set up of the SPI.  It runs at a
*         higher baud rate than the eeprom
*         SPI, and uses the interrupt.
*         However, use of the interrupt or
*         the higher baud rate is not necessary.
*
SUALT:
     		lda  	#%11010000    	; Interrupt with high baud rate.
     		sta  	SPCR          	; Set up the alternate SPI.
     		bclr 	CSLINE,CSPORT 	; ensure de-selection.
     		cli                	; Allow SPI interrupt.
     		rts

***********************************************
* CLRSPI - This sets the SPI to the reset
*          condition.
*
CLRSPI:
     		clr  	SPCR  		; Shut off SPI.
     		clr  	SPSR  		; Zero status register.
     		rts

*********************************************;
* EESEND - sends a byte through the SPI to
*          the serial eeprom and receives
*          a byte from the serial eeprom
*
* INPUTS    - accumulator, send to SPI
* OUTPUTS   - accumulator, response from SPI
*
* DESTROYS  - Accumulator
*
EESEND:
        	sta 	SPIDAT    		; Byte to send is in accumulator.
        	brclr	7,SPSR,$  		; Should loop 3 times.
        	lda   SPIDAT    		; Bring in what SPI has received.
        	rts

*********************************************;
* SENDADR - Sends two bytes to the eeprom through
*           the SPI.
*           A code is read in X to determine the
*           command sent to the eeprom.  They cross
*           as follows:
*             0 = WRITE ALL
*             1 = ERASE ALL
*             2 = ERASE
*             3 = WRITE
*             4 = EWDS
*             5 = EWEN
*           If the command is ERASE or WRITE the
*           Eeprom address is included
*           else only the command is included.
*
* INPUTS    - Number for the command in "X"
* OUTPUTS   - none
* DESTROYS  - Accumulator and "X"
*
SENDADR:
     		cpx  	#6    		; Ensure the value is in bounds.
     		bcc  	ABSEN 		; If invalid, exit.
     		txa        			; Copy command code to A.
     		lslx       			; Scale X input as a word pointer.
     		lsra       			; If command code is 0, 1, 4, or 5
     		lsra       			; an address is not to be sent.
     		bset 	CSLINE,CSPORT   	; Select the eeprom here.
     		bcc  	SEN2     		; Jump to code which will not send 	
					; address.
     		lda  	cmdlst,x 		; Bring in first byte of proper 	
					; command.
     		jsr  	EESEND   		; Send that first byte of command 	
					; out SPI.
     		lda  	ee_addr  		; Bring in eeprom address.
    	 	inca          		; Increment it for the next pass.
     		sta  	ee_addr  		; Update eeprom address for next 	
					; pass.
     		deca          		; Decrement eeprom address for this 
						; pass.
     		and  	#MASK    		; Mask off non-address bits.
     		ora  	cmdlst+1,x 		; OR second part of command with 	
					; address.
     		bra  	ADRDONE    		; Re-join the paths of this routine.

SEN2:                 			; No address send starts here.
     		lda  	cmdlst,x    	; Bring in first byte of proper 	
					; command.
     		jsr  	EESEND      	; Send that first byte of command 	
					; out SPI.
     		lda  	cmdlst+1,x  	; Bring in second byte of proper 	
					; command.

ADRDONE:
     		jsr  	EESEND      	; Send out the second part of 	
					; command with or without address.
ABSEN:         				; Branch around invalid commands 	
					; comes here.
     		rts

*
*********************************************;
* EWEN - This subroutine enables erase and write
*        operations.  It in effect unlocks the
*        eeprom so that its cells may be
*        changed.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - Accumulator and "X"
*
EWEN:
     		ldx  	#$05      		; Bring in 5 for EWEN.
     		jsr  	SENDADR   		; Interpret command 5 as EWEN.
     		bclr 	CSLINE,CSPORT  	; Release the eeprom.
     		rts

*********************************************;
* EWDS - This subroutine disables erase and
*        write operations so that data cannot be
*        inadvertently corrupted.  It in effect
*        locks the eeprom so that its cells
*        cannot be changed.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - Accumulator and "X"
*
EWDS:
     		ldx  	#$04      		; Bring in 4 for EWDS.
     		jsr  	SENDADR   		; Interpret command 4 as EWDS.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		rts

*********************************************;
* CK_CLR - This subroutine checks the status
*          of eeprom operation.  If the eeprom
*          is busy the routine loops until the
*          previous eeprom operation becomes
*          ready.  It also checks the SPI to
*          ensure that the SPI is interrupt
*          operating.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - Accumulator.
*
CK_CLR:
     		brset	CSLINE,CSPORT,NCLR ; Selected?
     		brset m_to_pr,flag,NCLR	 ; Programming?
     		bra  	READY 		; All Clear.

NCLR:      					; Not Clear, something going on.
     		cli        			; Ensure interrupts are on.
     		lda  	SPCR  		; Bring in SPI control register
     		cmp  	#SPIIRON 		; Check SPI control reg.
     		beq  	CK_LOOP  		; If OK do not re- set up.
     		jsr  	SUSPIR  		; Re-set up the SPI.

CK_LOOP:
     		brset CSLINE,CSPORT,$ 	; Loop until eeprom de-selected.
     		brset m_to_pr,flag,$  	; Loop until eeprom is free.

READY:
     		rts

*********************************************;
* ERAL - This subroutine erases the entire
*        eeprom.  An erased cell will put a high
*        level on the DO line when read, but
*        due to inverting in READ, the result
*        will arrive as 0x00 in 6805 memory.
*        ERAL calls EWEN to allow erasure.
*
* INPUTS    - none
* OUTPUTS   - none
* DESTROYS  - all contents of eeprom
*
ERAL:
     		jsr  	SUSPI  		; Set up the SPI for polling.
     		jsr  	EWEN   		; Open the eeprom.
     		ldx  	#$01  		; 1 is the command list location for 
						; erase all.
     		jsr  	SENDADR 		; Interpret command 1 as ERAL.
     		bra  	SIRXIT 		; Set interrupt and exit.

**********************************************
*  writing - The following is used to
*            write-all form 16 configured
*            eeproms.
*
#MACRO 	WRAL16
     		lda  	block_to_go  	; Bring back in the high byte to 	
					; write.
     		coma              	; Compliment it.
     		jsr  	eesend       	; Send it to the eeprom for 		
					; programming.
#MACROEND

********************************************;
* WRAL - In FORM8 eeproms this subroutine
*        writes the byte in the accumulator to
*        every byte of the Eeprom.  In FORM16
*        eeproms the accumulator is written to
*        the most significant byte the X
*        register is written to the less
*        significant byte.
*
* INPUTS    - Accumulator ( and X for FORM16)
* OUTPUTS   - none
* DESTROYS  - Accumulator, X, ee_addr .
*
*
WRAL:
     		sta  	mem_addr     	; Store low byte to be written.
     		stx  	block_to_go  	; Store high byte to be written.
     		jsr  	SUSPI        	; Set up the SPI for polling.

     		jsr  	EWEN    		; Open the eeprom for writing.
     		clrx         		; 0 is code for Write all.
     		jsr  	SENDADR 		; Interpret Command 0 as Write All.
     		lda  	mem_addr  		; Bring back data for sending.
     		coma           		; Complement for writing.
     		jsr  	eesend    		; Send byte to eeprom for writing.

#IF 		9346FORM16
     		WRAL16    			; Write upper byte of a 16 bit wide 
						; eeprom.
#ENDIF

#IF 		9356FORM16
     		WRAL16    			; Write upper byte of a 16 bit wide 
						; eeprom.
#ENDIF

#IF 		9366FORM16
     		WRAL16    			; Write upper byte of a 16 bit wide 
						; eeprom.
#ENDIF
     		bra  	SIRXIT 		; Set interrupt and exit.

*********************************************;
* ERASE - This subroutine Erases an eight
*         cell byte or 16 cell word in the
*         Eeprom.  The address of the cell is
*         located in the accumulator.  The
*         accumulator is returned unchanged.
*
* INPUTS    - The following memory locations
*             set up as follows.
*         ee_addr   -> contains the absolute
*                      address of where the
*                      erasure will start in
*                      the eeprom.
*         mem_addr  -> (not used)
*
*         block_to_go  -> The length of the block,
*                      1 writes one byte,
*                      0 writes none.
* OUTPUTS   - none
* DESTROYS  - ee_addr , mem_addr , block_to_go,
*             Accumulator and X
*
*
ERASE:
     		jsr  	SUSPI        	; Set up the SPI for polling.
     		lda  	block_to_go  	; Bring in the length of block to be 
						; erased.
     		beq  	WRDONE       	; When the block is 0 use the same 	
					; finish as write.
     		deca              	; one less in the block to go.
     		sta  	block_to_go  	; Update the block to go.
     		brset m_to_pr,flag,NOTER1 ; Check for first pass.
    	 	jsr  	EWEN         	; Open the eeprom for erasure.
     		bset 	m_to_pr,flag 	; Set programming function flags.
    		bclr 	WR,flag      	; Clear writing flag.

NOTER1:
     		ldx  	#$02         	; Erase is selection 2.
     		jsr  	SENDADR      	; Send address to erase.
     		bra  	SIRXIT 		; Set interrupt and exit.

**********************************************
*  writing - The following is used to write
*            form 16 configured eeproms.
*
#MACRO 	WR_BYTE
     		lda  	,x    		; X still points to the next byte,
     		coma       			; bring it into the acc. and 		
					; complement.
     		jsr  	EESEND  		; Send the byte.
     		incx         		; Increment to point to the next 	
					; byte.
     		stx  	mem_addr		; Store updated pointer for the next 
						; pass.
     		lda  	block_to_go  	; Bring in the remaining block 	
					; length.
     		beq  	done16       	; IF zero length we are done.
     		deca              	; IF not decrement the block length.
     		sta  	block_to_go  	; Update the block length.
done16:
#MACROEND

*********************************************;
* WRITE - This subroutine Writes a block of
*         eight cell bytes to the Eeprom.
*
* INPUTS    - The following memory locations
*             set up as follows.
*         ee_addr   -> contains the absolute
*                      address of where the
*                      data will start in
*                      the eeprom.
*         mem_addr  -> contains the absolute
*                      starting address of the
*                      block of memory
*                      to be written to eeprom.
*         block_to_go  -> The length of the block,
*                      1 writes one byte,
*                      0 writes none.
* OUTPUTS   - none
* DESTROYS  - ee_addr , mem_addr , block_to_go,
*             Accumulator and X
*
*
WRITE:
     		jsr  	SUSPI         	; Set up the SPI for sending data.
     		lda  	block_to_go   	; Bring in the block length left.
     		beq  	WRDONE        	; If zero block length we are 	
					; finished.
     		deca               	; Decrement the block length.
     		sta  	block_to_go   	; Update block length.
     		brset m_to_pr,flag,NOTWR1 ; Check for first pass.
     		jsr  	EWEN         	; IF first pass open the eeprom.
     		bset 	m_to_pr,flag 	; Set programming in process flags.
     		bset 	WR,flag      	; Set the writing flag.

NOTWR1:
     		ldx  	#$03        	; code for write is 3.
     		jsr  	SENDADR     	; Send the code for writing and ee 	
					; address.
     		ldx  	mem_addr    	; Bring in the memory address as 	
					; pointer.
     		lda  	,x          	; Bring in the byte to be written.
     		incx             		; Increment the pointer for next 	
					; pass.
     		stx  	mem_addr    	; Update memory address pointer.
     		coma             		; Complement the byte to be written.
     		jsr  	eesend      	; Send it to the eeprom to be 	
					; programmed into it.

#IF 		9346FORM16
     		WR_BYTE   			; Send the second byte to be 		
					; programmed if form 16.
#ENDIF

#IF 		9356FORM16
     		WR_BYTE   			; Send the second byte to be 		
					; programmed if form 16.
#ENDIF

#IF 		9366FORM16
     		WR_BYTE   			; Send the second byte to be 		
					; programmed if form 16.
#ENDIF

SIRXIT:   					; Put the eeprom into the busy ready 
						; mode.
     		bclr 	CSLINE,CSPORT 	; De-select the Eeprom
     		bset 	CSLINE,CSPORT 	; Re-Select the Eeprom
     		jsr  	SUSPIR  		; Set up SPI int on and lowest baud 
						; rate.
     		clra         		; Send Zeros out SPI.  Every 8 	
					; clocks the
     		sta  	SPIDAT  		; interrupt will fire and the eeprom 
						; will be checked for ready.
     		cli          		; Enable SPI interrupt.
     		rts

WRDONE:
        	bclr	m_to_pr,flag 	; no more to write.
        	jsr   EWDS         	; Write protect the eeprom.
        	rts

**********************************************
*  reading - The following is used to read
*            form 16 configured eeproms.
*
#MACRO 	RD_BYTE
     		jsr  	EESEND         	; read a byte from an addressed 	
					; eeprom.
     		ldx  	block_to_go    	; Read in the remaining block 	
					; length.
     		beq  	RD16END        	; Only store if remaining length is 
						; non Zero.
     		decx                	; Decrement the remaining block 	
					; length.
     		stx  	block_to_go    	; Update the remaining block length.
     		coma                	; Complement the byte read from the 
						; eeprom.
     		ldx  	mem_addr       	; Load the pointer with the address 
						; to place
                         		; the byte read from memory.
     		sta  	,x             	; Store the read byte to memory.
     		incx                	; Increment the memory address 	
					; pointer.
     		stx  	mem_addr       	; Update the memory address pointer.
RD16END:
#MACROEND

*
*********************************************;
* READ - This subroutine reads a block of
*        data out of the eeprom and places it
*        in a block of 6805 memory.  It has the
*        autosequence feature as an option.
*
* INPUTS    - The following memory locations
*             set up as follows.
*         ee_addr   -> contains the eeprom
*                      address where the data
*                      block starts.
*
*         mem_addr  -> contains the absolute
*                      starting
*                      address of the 6805
*                      memory block
*                      destination.
*
*         block_to_go  -> The length of the block,
*                         1 reads one byte,
*                         0 reads none.
* OUTPUTS   - a block of updated memory
* DESTROYS  - ee_addr , mem_addr , block_to_go,
*             Accumulator and X
*
*
READ:
     		jsr  		SUSPI      	; Set up the SPI for polling.

     		jsr  		EWDS       	; Write protect the eeprom as a
                     			; precaution. Anything worth reading
                     			; is worth protecting.

RDNLP:                 			; Loop for non-autosequenced read.
     		ldx  	block_to_go  	; Read in the remaining block 	
					; length.
     		tstx              	; Test length to see if done.
     		beq  	RDNDONE      	; IF done jump out of routine.
     		decx              	; decrement the block for this pass.
     		stx  	block_to_go  	; Store updated block length for 	
					; checking on the next pass.

#if 		9366FORM16
     		lda  	ee_addr   		; Bring in eeprom address
     		lsla           		; Place MS Bit in carry.
     		clra           		; Zero out the accumulator.
     		rola           		; MS Bit of ee_address is LS Bit of 
						; accumulator.
     		ora  	#READ1    		; Overlay read command #1.
#ELSEIF
     		lda  	#READ1    		; Bring in read command #1.
#ENDIF
     		lda  	#READ1 		; Bring in Read command #1.
     		tst 	SPSR   		; clean out the SPI status register.
     		tst  	SPIDAT 		; clean out the SPI receiver.
     		bset 	CSLINE,CSPORT 	; Select the Eeprom
     		jsr  	EESEND        	; Send READ command 1.
     		lda  	ee_addr       	; Bring in the eeprom address to be 
						; read.
     		and  	#MASK         	; Mask off non-valid bits of 		
					; address.
     		lsla               	; Shift left to accommodate read
                        		; transition clock.
     		ora  	#READ2  		; OR address with second part of 	
					; READ command.
     		jsr  	EESEND  		; Send READ2 | ee_address.
     		lda  	ee_addr 		; Bring in ee_address.
     		inca         		; Increment ee_address.
     		sta  	ee_addr 		; Update ee_address for next pass.
     		clra         		; Clear Acc. so a logic low is sent 
						; to eeprom.
    		jsr  	EESEND  		; Read first byte.
     		coma         		; Complement byte, all data is 	
					; complemented.
     		ldx  	mem_addr		; Bring in pointer for memory.
     		sta  	,x      		; Store the read byte in the memory 
						; location pointed to.
     		incx         		; Increment the memory pointer for 	
					; next pass.
     		stx  	mem_addr		; Update memory pointer.

#IF AUTOSEQ    				; If an autosequence eeprom a 	
					; smaller more efficient loop may be 	
					; used.
WRLOOP:
     		RD_BYTE            	; Read byte and store.
     		bne  	WRLOOP        	; IF block to read is not zero, read 
						; more.
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		bra  	RDNDONE       	; Branch to out of routine.  None of 
						; the rest of the code is used if 	
					; the eeprom is autosequence.
#ENDIF

#IF 		9346FORM16
     		RD_BYTE            	; Read byte and store.
#ENDIF

#IF 		9356FORM16
     		RD_BYTE            	; Read byte and store.
#ENDIF

#IF 		9366FORM16
     		RD_BYTE            	; Read byte and store.
#ENDIF
     		bclr 	CSLINE,CSPORT 	; deselect the Eeprom
     		bra  	RDNLP   		; Branch always, block check is done 
						; above.

RDNDONE:
     		clr  	flag  		; Clear the eeprom flag.
     		jsr  	SUALT 		; Set up alternate SPI.
     		rts


*################################################
* SPI Interrupt handler
*
* SPI interrupt handler is only used to assess the
* eeprom's ready condition during erasure, or writing.
SPI:
     		brclr	CSLINE,CSPORT,SPIALT ; IF eeprom is not selected,
                              	; then some other interrupt
                                  	; driven SPI service must be
                                  	; active.  Jump there.
     		tst	SPSR            	; Test status register to reset 	
					; interrupt.
     		lda   SPIDAT          	; Read data, to reset interrupt, 	
					; too.
     		beq   NOTREADY        	; If Zero, the eeprom is not ready.
     		bclr  CSLINE,CSPORT   	; IF ready, deselect for next 	
					; command.
     		brclr m_to_pr,flag,FINISH  ; Single program cycle, go to 	
					; end.
     		lda   block_to_go  	; A WRITE is in process, check for 	
					; done.
     		beq   FINISH       	; No more to write, go to end.
     		brset WR,flag,WRF  	; erase or write?
     		jsr   ERASE        	; If erase, and not finished, erase 
						; another.
     		bra   SPIDONE      	; Done for now.

WRF:
     		jsr 	WRITE        	; IF WRITE, and not finished, write 
						; another.
     		bra 	SPIDONE      	; Done for now.

NOTREADY:             			; IF eeprom not ready send more 	
					; clocks.
     		clra             		; Zero the accumulator to send 	
					; eeprom logic low.
     		sta  	SPIDAT   		; Send clocks to the eeprom using 	
					; SPI.

SPIDONE:
     		rti

FINISH:
     		bclr	m_to_pr,flag  	; Clear the more to program flag.
     		jsr   CLRSPI        	; Reset SPI.
     		jsr   SUALT         	; Set up the SPI for an alternate
                           		; handler.
SPIALT:
     		tst   SPSR  		; This is filler, put the
     		lda   #$69  		; other SPI handler here
     		sta   SPIDAT		; if it will be interrupt
     		rti           		; driven.

*###############################################

*********************************************;
* START - Sample calling of routines.
*
BSTART  	EQU  	0
BL_LEN  	EQU  	$80  			; Length of block for examples.

STARTRD:
     		jsr  	SETUP    		; Flags must be cleared on system 	
					; start up.
     		jsr  	IDIO     		; Set up lines idiosyncratic of this 
						; hardware.
     		jsr  	CK_CLR   		; Ensure the eeprom is free.
    		lda   #BSTART    		; Start reading eeprom at
                     			; address BSTART.
    		sta   ee_addr    		; Place that address in memory so 	
					; program can get it.
    		lda   #data      		; Get the location of the lowest 	
					; vector block.
    		sta   mem_addr   		; Place it in memory so the program 
						; can get it.
    		lda   #BL_LEN    		; Length of block to read.
    		sta   block_to_go		; Place it in memory so the program 
						; can get it.
    		jsr  	READ     		; Read whatever is in the eeprom. 
    		bra  	$        		; jump to this location
                  			; (do nothing else).

STARTWR:
     		jsr  	SETUP    		; Flags must be cleared on system 	
					; start up.
     		jsr  	IDIO     		; Set up lines idiosyncratic of this 
						; hardware.
     		WRBLOCK   BSTART,data,BL_LEN  ; See macro at the beginning.
     		jsr  	CK_CLR  		; Protect memory during write.

* Zero out all memory as a test.
    		lda  	#$ff  		; Place #ff in highest
     		sta  	$ff   		; place in lower RAM
     		ldx  	#$50  		; Start of lower RAM.
     		clra       			; Acc = 0 to be written to lower 	
					; RAM.

LOOP2:
     		sta  	,x    		; Place the Zero in Acc in memory 	
					; pointed to.
     		incx       			; Increment memory pointed to.
     		brset	1,$ff,LOOP2   	; When the last byte goes to 0, 	
					; done.
     		bra  	STARTRD        	; Read back from eeprom, should be 	
					; the same.

STARTERAL:
     		jsr  	SETUP    		; Flags must be cleared on system 	
					; start up.
     		jsr  	IDIO     		; Set up lines idiosyncratic of this 
						; hardware.
     		jsr  	CK_CLR   		; Ensure the eeprom is free.
     		jsr  	ERAL     		; Erases the entire serial eeprom
     		bra  	$        		; (do nothing else).

STARTWRL:
     		jsr  	SETUP    		; Flags must be cleared on system 	
					; start up.
     		jsr  	IDIO     		; Set up lines idiosyncratic of this 
						; hardware.
    		lda  	#$a5  		; (write $a5 to form 8 eeprom.)

#IF 		9346FORM16
    		ldx  	#$c3  		; write $a5c3 to form 16 eeprom.
#ENDIF

#IF 		9356FORM16
    		ldx  	#$c3  		; write $a5c3 to form 16 eeprom.
#ENDIF

#IF 		9366FORM16
    		ldx  	#$c3  		; write $a5c3 to form 16 eeprom.
#ENDIF
    		jsr  	CK_CLR  		; Ensure the eeprom is free.
    		jsr  	WRAL    		; 0xa5 to all memory locations in
                 				; the eeprom. ($a5c3 to 16 bit form)
*     	bra  	$        		; (do nothing else).

STARTERSE:
    		jsr  	SETUP    		; Flags must be cleared on system 	
					; start up.
    		jsr  	IDIO   		; Set up lines idiosyncratic of this 
						; hardware.
    		lda  	#$05    		; ee_address to start block erasure.
    		sta   ee_addr		; Place it in memory so the program 
						; can get it.
    		lda   #3     		; Length of block to erase
    		sta   block_to_go		; Place it in memory so the program 
						; can get it.
    		jsr  	CK_CLR   		; Ensure the eeprom is free.
    		jsr  	ERASE    		; Erases memory location 5+ of the
                  			; eeprom.
     		bra  	$        		; (do nothing else).


IRQ:   					; External interrupt.
     		jsr  	IDIO 			; Should never get here.
     		rti

	ORG 	VECTORS

VECSPI:  	fdb   SPI       		; SPI VECTOR
VECSCI:  	fdb   STARTRD   		; SCI VECTOR
VECTMR:  	fdb   STARTRD   		; TIMER VECTOR
VECIRQ:  	fdb   IRQ       		; IRQ VECTOR
VECSWI:  	fdb   STARTWR   		; SWI VECTOR
VECRST:  	fdb   STARTRD   		; START VECTOR

	


