******************************************************************************
******************************************************************************
*                                                                            *
*  Main Routine SCI_01 - SCI Software Transmit/Receive Routines              *
*                                                                            *
******************************************************************************
*                                                                            *
* File Name: SCI_01.RTN                   Copyright (c) Motorola 1995        *
*                                                                            *
* Full Functional Description of Routine Design:                             *
*   Program flow:                                                            *
*       Reset:  Call init to intialize port pins                             *
*               Call get_char to receive a byte of data                      *
*               Light frame error LED if frame error occured                 *
*               Light noise LED if frame error occured                       *
*               Call put_char to transmit the received byte of data          *
*               Loop back to get_char call (endless loop)                    *
*                                                                            *
******************************************************************************
*                                                                            *
*        Part Specific Framework Includes Section                            *
*                                                                            *
******************************************************************************
#INCLUDE 'H705J1A.FRK'                         ; Include the equates for the
                                               ; HC705J1A so all labels can
                                               ; be found.

******************************************************************************
*                                                                            *
*        MOR Bytes Definitions for the Main Routine                          *
*                                                                            *
******************************************************************************
		org	MOR 
                fcb     $00

******************************************************************************
*                                                                            *
*        Equates and RAM Storage                                             *
*                                                                            *
******************************************************************************
*** I/O Pin Equates:
serial_port     equ     $01                    ; port used for serial port pins
status_port     equ     $00                    ; port used for driving LED's.
noise           equ     4                      ; pin # for noise LED
frame           equ     5                      ; pin # for frame LED
rxd             equ     0                      ; pin # for receive data pin
txd             equ     1                      ; pin # for transmit data pin

*** Program Constant Equates:                  ; Baud rate select table:
BAUD_SEL        equ     $08                    ; BAUD_SEL   4MHz osc    2MHz osc
                                               ;    $04      19.2k         9600
                                               ;    $08       9600         4800
                                               ;    $10       4800         2400
                                               ;    $20       2400         1200
                                               ;    $40       1200          600
                                               ;    $80        600          300
*** RAM variable allocation:
                org     RAM
char            rmb     1                      ; data register for sci
count           rmb     1                      ; temp storage variable

******************************************************************************
* main - example program that continually echoes back received characters.   *
*                                                                            *
* input cond.     - reset                                                    *
* output cond.    - none (infinite loop)                                     *
* stack used      - 4 bytes                                                  *
* variables used  - none                                                     *
* ROM used        - 28 bytes                                                 *
******************************************************************************
                org     ROM                    ; start at the top of ROM
main            rsp                            ; reset the stack pointer
                jsr     init                   ; initialize port pins

main_loop       jsr     get_char               ; receive one byte of data
                                               ; from rxd pin

                bcc     no_frame_error         ; branch if no noise occured
                bclr    frame,status_port      ; turn on frame LED
                bra     continue               ; don't check for noise --
                                               ; it's undefined
no_frame_err    bset    frame,status_port      ; turn off frame LED

                bhcs    noise_error            ; branch if noise occured
                bset    noise,status_port      ; turn off noise LED
                bra     continue               ; skip next line of code
yes_noise_err   bclr    noise,status_port      ; turn on noise LED

continue        jsr     put_char               ; transmit the received byte
                bra     main_loop              ; and prepare for next
                                               ; reception.

******************************************************************************
* init - initialize port pins for sci operation and for driving LEDs;        *
*        called by main                                                      *
*                                                                            *
* input cond.     - none                                                     *
* output cond.    - TXD = output initialize to 1, RXD = input, noise LED =   *
*                   off, frame LED = off.                                    *
* stack used      - 0 bytes                                                  *
* variables used  - none                                                     *
* ROM used        - 15 bytes                                                 *
******************************************************************************
init            bset    txd,serial_port        ; init txd = 1
                bset    txd,serial_port+4      ; txd = output

                bclr    rxd,serial_port+4      ; rxd = input

                bset    noise,status_port      ; noise LED = off
                bset    noise,status_port+4    ; noise = output

                bset    frame,status_port      ; frame LED = off
                bset    frame,status_port+4    ; frame = output
                rts                            ; exit (init)

******************************************************************************
* get_char - receive one byte of data from RXD pin; called by main           *
*                                                                            *
* input cond.     - RXD pin defined as an input pin                          *
* output cond.    - char contains received data; X,ACC undefined;            *
*                   half carry = 1 (frame occured) or 0 (no frame error);    *
*                   carry = 1 (noise and/or frame error occured) or 0        *
*                   (no noise).                                              *
* stack used      - 2 bytes                                                  *
* variables used  - char: storage for received data (1 byte)                 *
*                   count: temporary storage (1 byte)                        *
* ROM used        - 63 bytes                                                 *
******************************************************************************
get_char        lda     #8                     ;[2] receiving 8 data bits
                sta     count                  ;[4] store value into RAM
                clrx                           ;[3] used to store noise data

get_start_bit   brclr   rxd,serial_port,*      ;[5] wait until rxd=1
                brset   rxd,serial_port,*      ;[5] wait for start bit

                lda     #BAUD_SEL-3            ;[2] prepare for 1/2 bit delay
                bsr     delay_13a              ;[13a+12] execute delay routine
                bsr     get_bit                ;[39] sample start bit
                lsra                           ;[3] noise bit -> carry;
                                               ;    acc=filtered start bit
                bne     get_start_bit          ;[3] if false start, start over
                tsta                           ;[3] for timing purposes only
                tsta                           ;[3] for timing purposes only

                lda     #2*(BAUD_SEL-2)        ;[2] prepare for 1 bit delay
                bsr     delay_13a              ;[13a+12] execute delay routine

get_data_bits   bsr     get_bit                ;[39] sample data bit
                rora                           ;[3] noise bit -> carry
                rorx                           ;[3] carry -> noise data reg
                rora                           ;[3] filtered data bit -> carry
                ror     char                   ;[5] carry -> char
                lda     #2*(BAUD_SEL-3)        ;[2] prepare for 1 bit delay
                bsr     delay_13a              ;[13a+12] execute delay routine
                tsta                           ;[3] for timing purposes only
                dec     count                  ;[5] bit received, dec count
                bne     get_data_bits          ;[3] loop if more bits to get

get_stop_bit    bsr     get_bit                ;[39] sample stop bit
                lsra                           ;[3] noise bit -> carry
                                               ;    acc=filtered stop bit
                sta     count                  ;[4] store stop bit in count
                bcc     yes_noise              ;[3] if noise, then branch

                txa                            ;[2] noise data -> acc
                eor     char                   ;[3] XOR noise with char,
                beq     no_noise               ;[3] and if result=0,
                                               ;    then no noise in data
                                               ;    reception

yes_noise       lda     #$08                   ;[2] set noise bit (half carry)
                add     #$08                   ;[2] by adding $8 to $8

no_noise        lda     count                  ;[3] retrieve stop data bit,
                coma                           ;[3] complement it,
                lsra                           ;[3] and shift it into carry
                                               ;    for frame error bit
                rts                            ;[6] exit (get_char)

******************************************************************************
* get_bit - receive one bit of filtered data and noise info; called by       *
*            get_char                                                        *
*                                                                            *
* input cond.     - RXD pin defined as an input pin                          *
* output cond.    - ACC = 000000dn, where d = filtered data, n = noise info  *
* stack used      - 0 bytes                                                  *
* variables used  - none                                                     *
* ROM used        - 17 bytes                                                 *
******************************************************************************
get_bit         clra                           ;[3] used to add sampled bits
                brset   rxd,serial_port,samp_1 ;[5] sample 1st bit into carry
samp_1          adc     #0                     ;[3] add it to acc
                brset   rxd,serial_port,samp_2 ;[5] sample 2nd bit into carry
samp_2          adc     #0                     ;[3] add it to acc
                brset   rxd,serial_port,samp_3 ;[5] sample 3rd bit into carry
samp_3          adc     #0                     ;[3] add it to acc
                rts                            ;[6] exit (get_bit)

******************************************************************************
* put_char - transmit data byte in char out onto TXD pin; called by main     *
*                                                                            *
* input cond.     - TXD pin defined as an output pin and TXD = 1;            *
*                   char contains byte to be tranmitted.                     *
* output cond.    - X,ACC,char =  undefined;                                 *
* stack used      - 2 bytes                                                  *
* variables used  - char: storage for transmitted data (1 byte)              *
* ROM used        - 31 bytes (35 if sending two stop bits)                   *
******************************************************************************
put_char        ldx     #9                     ;[2] be sending 8 data bits
                clc                            ;[2] clear carry for start bit

put_data_bits   bcc     send_0                 ;[3] if carry<>0, then
                bset    txd,serial_port        ;[5]    send out a 1
                bra     jmp_bit                ;[3]    finished sending a 1
send_0          bclr    txd,serial_port        ;[5] else send a 0
                bra     jmp_bit                ;[3]    finished sending a 0
jmp_bit         lda     #2*(BAUD_SEL-1)-1      ;[2] prepare for a 1 bit delay
                bsr     delay_13a              ;[13a+12] execute delay routine
                tsta                           ;[3] for timing purposes only
                ror     char                   ;[5] get next data bit to send
                decx                           ;[3] one bit sent, so dec count
                bne     put_data_bits          ;[3] loop if more bits to send

put_stop_bit    nop                            ;[2] for timing purposes only
                bset    txd,serial_port        ;[5] send out a one
                lda     #2*(BAUD_SEL-1)        ;[2] prepare for a 1 bit delay
                bsr     delay_13a              ;[13a+12] execute delay routine

* add the next two lines to guarantee sending two stop bits:
*               lda     #2*(BAUD_SEL-1)+1      ;[2] prepare for a 1 bit delay
*               bsr     delay_13a              ;[13a+12] execute delay routine

                rts                            ;[6] exit (put_char)

******************************************************************************
* delay_13a - delay for 13*ACC + 12 cycles; called by get_char and put_char  *
*                                                                            *
* input cond.     - ACC set to appropriate value (13*ACC + 12 cycles)        *
* output cond.    - ACC = 0                                                  *
* stack used      - 0 bytes                                                  *
* variables used  - none                                                     *
* ROM used        - 7 bytes                                                  *
******************************************************************************
delay_13a       nop                            ;[2] this is a 13-cycle loop
                nop                            ;[2]
                tsta                           ;[3]
                deca                           ;[3] decrement loop count
                bne     delay_13a              ;[3] loop if count not zero
                rts                            ;[6] exit (delay_13a)


******************************************************************************
*                                                                            *
*        Interrupt and Reset vectors for Main Routine                        *
*                                                                            *
******************************************************************************
*** reset vectors:
                org     RESET
                fdb     main

