;-----------------------------------------------------------------------------
;              FREQUENCEMETRE A LIASON SERIE ASYNCHRONE
;                                  
;                 microcontroleur ST6 de SGS-THOMSON
;-----------------------------------------------------------------------------

;
; Vitesse de transmission determinee par le temps d'execution des instructions
;
;                                      13
; Temps d'un cycle instruction Tcy = -------
;                                    Fquartz
;
; Toutes les instructions necessite 4 cycles Tcy sauf :
;
; - JRxx (sauts relatifs), RET, RETI, STOP, WAIT   :   2 Tcy
; - JRR,JRS ( sauts relatifs selon etat d'un bit ) :   5 Tcy
;
;                                         1
; Temps de transmission d'un bit Tbit =  ----
;                                        baud
;
;                  ------------------------------------
;------------------!         E M I S S I O N          !-----------------------
;                  ------------------------------------
;                                                              Fquartz
; START BIT:  Tbit = (6 * Tempbit0 + 30) * Tcy  ==> Tempbit0 = ------- - 5
; ----------                                                   78*Baud
;
;                                                              Fquartz
; BIT :       Tbit = (6 * Tempbit1 + 48) * Tcy  ==> Tempbit1 = ------- - 8
; ---                                                          78*Baud
;
;                                                              Fquartz
; STOP  BIT:  Tbit = (6 * Tempbit2 + 04) * Tcy  ==> Tempbit2 = ------- - 0,7
; ----------                                                   78*Baud
;
; tableau recapitulatif pour differentes vitesses de transmission avec
; un quartz de 6,000 Mhz
;
;      ------------------------------------------------
;      | Baud   |  tempbit0  |  tempbit1  |  tempbit2 |
;      |--------|------------|------------|-----------|
;      | 1200   |    59      |    56      |    64     |
;      ------------------------------------------------
;
;
;                  ------------------------------------
;------------------!         R E C E P T I O N        !-----------------------
;                  ------------------------------------
;
;                 Tbit                                          Fquartz
; 1/2 START BIT:  ---- = ( 15* Tempbit0) * Tcy  ==> Tempbit0 = -----------
; -------------     2                                           390 * Baud
;
;                                                              Fquartz
; BIT 0 :     Tbit = (10* Tempbit1 + 24) * Tcy  ==> Tempbit1 = ------- - 2,4
; ---                                                          130*Baud
;
;                                                              Fquartz
; BIT 1  7 : Tbit = (10* Tempbit1 + 47) * Tcy  ==> Tempbit1 = ------- - 4,7
; ---                                                          130*Baud
;
; cette dernire est utilise pour dterminer Tempbit1
;                                                              Fquartz
; STOP  BIT:  Tbit = (10* Tempbit2 + 08) * Tcy  ==> Tempbit2 = ------- - 0,8
; ----------                                                   130*Baud
;
; tableau recapitulatif pour differentes vitesses de transmission avec
; un quartz de 6,000 Mhz
;
;      ------------------------------------------------
;      | Baud   |  tempbit0  |  tempbit1  |  tempbit2 |
;      |--------|------------|------------|-----------|
;      | 1200   |    13      |    35      |    38     |
;      ------------------------------------------------
;
;                __________________________________________________
;                !            INTERRUPTION TIMER                  !
;----------------!  ( base de temps de comptage des priodes )    !-----------
;                --------------------------------------------------
;
;             12
; Ttimer =  ------- * Prescaler * Timer ,
;           FQuartz
;
; Avec Prescaler = 32 , FQuartz = 6MHz et Timer = 125 on obtient une base
; de temps de 8ms , donc 1s = 125 * Timer. Cette interruption contrle le
; temps pendant lequel sont comptes les priodes.
;
; Le comptage est assur par un compteur externe dont les 8 bits LSB sont
; utilis, puis les dbordements sont compts par l'interruption sur front
; descendant de PC4.
;
;****************************************************************************


           .title "frequencemetre"
           .vers "st6225"
           .input "6225_reg.asm"
           .w_on

;------------------------------------------
;       definition des constantes
;------------------------------------------

chien      .equ 0FEh            ; valeur du chien de garde
tbit0_e    .equ 59              ; tempo start bit  ( pour 1200 bauds )
tbit1_e    .equ 56              ; tempo bit        ( pour 1200 bauds )
tbit2_e    .equ 64              ; tempo stop bit   ( pour 1200 bauds )
tbit0_r    .equ 13              ; tempo 1/2 start  ( pour 1200 bauds )
tbit1_r    .equ 35              ; tempo bit        ( pour 1200 bauds )
tbit2_r    .equ 38              ; tempo stop bit   ( pour 1200 bauds )
tx         .equ 7               ; PC7 = TX
rx         .equ 6               ; PC6 = RX

;Bits du registre TSCR
tmz        .equ 7            ; mis a 1 quand Timer(TCR) passe a 0.
eti        .equ 6            ; eti=1  <=> autorisation interruptions Timer
tout       .equ 5            ; defini le mode de fonctionnement du Timer,
dout       .equ 4            ; conjointement avec le bit dout.
psci       .equ 3            ; psci=1  <=> fonctionne; psci=0 <=> TCR gele.

ValTimer   .equ 125
Val125     .equ 125
TimerMode  .equ 01011101b    ; Division par 32, IT timer autorises

Raz        .equ 0            ; PB0 = remise  zro du compteur
Enable     .equ 1            ; PB1 = autorisation du comptage

;------------------------------------------
; definition des variables en ram $84  $BF
;------------------------------------------

BaseRam    .def 084h            ; dbut de la zone RAM utilisateur
copypc     .def BaseRam         ; pour maniplation bits du port C
caracter   .def copypc+1        ; Sauvegarde caractere emis./recep.
Temps1     .def caracter+1      ; Pour base de temps 1s
RegFreqT   .def Temps1+1        ; 8 bits lsb du comptage des priodes
RegFreq0   .def RegFreqT+1      ; registre comtage (pour comptage sur 16 bits)
RegFreq1   .def RegFreq0+1      ; registre comtage (extension sur 24 bits)
Status     .def RegFreq1+1      ; registre d'tat du frquencemetre
                                ;      Bit 0 : 1 -> mesure en cours
                                ;      Bit 1 : 1 -> dbordement
CPL         .def Status+1       ; registre de comparaison 16bits
CPH         .def CPL+1
CCT0        .def CPH+1          ; registre de recopie du compteur
CCT1        .def CCT0+1
ASCII0      .def CCT1+1         ; registres pour valeur ASCII du compteur
ASCII1      .def ASCII0+1
ASCII2      .def ASCII1+1
ASCII3      .def ASCII2+1
ASCII4      .def ASCII3+1
NBCAR       .def ASCII4+1       ; contient nombre de chiffres


;------------------------------------------
;           debut du programme
;------------------------------------------

           .org 080h

; definition de la table des bits de masquage ncessaires  la liason srie

maskbit1   .byte   1 , 2 , 4 , 8 , 16 , 32 , 64 , 128
maskbit0   .byte   0FEh , 0FDh , 0FBh , 0F7h , 0EFh , 0DFh , 0BFh , 7Fh

TABP10     .equ $
           .word  10000  ; 10^4     
           .word  01000  ; 10^3
           .word  00100  ; 10^2
           .word  00010  ; 10^1
           .word  00001  ; 10^0

; initialisation  de ports
;      PC7        = sortie TX etage push-pull, initialise a 1
;      PC5 et PC6 = entree sans rsistance de pull-up sans I.T.
;                   ( PC6 = RX , PC5 inutilise )
;      PC4        = entre avec rsistance de pull-up avec I.T. sur front bas
;                   ( dtection du dbordement du compteur )
;      PA         = entre sans rsistance de pull-up sans I.T.
;                   ( lecture des 8bits LSB du compteur )
;      PB         = sorties symtriques push-pull
;                   ( PB0 = raz comteur, PB1 = autorisation comptage )


init_pgm   ldi wdr,chien
           ldi pcdir,10000000b
           ldi pcopt,10010000b
           ldi pc,11000000b
           ldi copypc,11100000b
           clr padir
           clr paopt
           ldi pa,0FFh
           ldi pbdir,0FFh
           ldi pbopt,0FFh
           clr pb
           clr tscr             ; inhibe Timer pour le moment
           clr Status           ; frequencemetre  l'arret
           reti
           ldi ior,00010000b    ; autorise IT timer et PC4 sur front descend.
           set Raz,pb           ; RAZ compteur 8bits pour mesure

Synchro    call recoit          ; Attend rception du caractre 'F' pour
           cpi a,'F'            ; effectuer une mesure de frquence
           jrnz TestL           ; sinon, si caractre 'R', emet 'O'
           jp Mesure
TestL      cpi a,'R'
           jrnz Synchro
           ldi a,'O'
           call emet
           jp Synchro

Mesure     clr RegFreq1         ; 
           clr RegFreq0
           ldi Temps1,Val125    ; init. pour une priode de 1s
           clr psc              ; remise  zro du prdiviseur
           ldi tcr,ValTimer     ; init. pour 8ms
           ldi tscr,TimerMode   ; Lance base de temps
           set Enable,pb        ; autorise le comtage (mise  1 de PB1)
           res Raz,pb           ; RAZ inactive
           set 0,Status         ; indique mesure en cours
attend     ldi wdr,chien
           jrs 0,Status,attend  ; attend la fin de la mesure

resultat   ld a,pa              ; rcupre les 8 bits lsb
           ld RegFreqT,a
           ld a,RegFreq1        ; si RegFreq1=0, pas de dpassement de 
           jrz NotOver          ; calibre.
           jp OverFlow          ; sinon, dpassement...
NotOver    call ConvAscii       ; conversion 16 bits en BCD cod ASCII
           call AsciiOut        ; transmission des 5 chiffres
           set Raz,pb           ; RAZ compteur 8bits pour mesure
           jp Synchro           ; relance nouvelle mesure synchronise
OverFlow   ldi a,'D'            ; envoit la lettre 'D' pour dpassement
           call emet
           set Raz,pb           ; RAZ compteur 8bits pour mesure
           jp Synchro           ; relance nouvelle mesure synchronise


;---------------------------------------------------------------
; routine d'emission d'un caractere par liaison serie asynchrone
; le caractere a transmettre doit etre charge dans l'accu A
;---------------------------------------------------------------

emet       ld caracter,a        ; sauvegarde caractere a emettre
           ldi drwr,maskbit1.w  ; pointe, via reg. X, la table des masques
           ldi x,maskbit1.d
           ldi v,tbit0_e        ; emission bit de start ( tempo reg. V )
           res tx,copypc        ; force TX = 0 en conservant la configuration
           ld a,copypc          ; des entrees/sorties
           ld pc,a
emet_0     dec v                ; tempo start bit
           jrnz emet_0
           ldi w,8              ; initialise le compteur de bits a 8
emet_1     ldi wdr,chien        ; relance le chien de garde
           ld a,caracter        ; rappel le caractere a transmettre
           and a,(x)            ; test l'etat du bit courant
           jrnz emet_3          ; selon etat met TX a 1 ou 0 logique
           res tx,copypc
           ld a,copypc
           ld pc,a
           jp emet_4
emet_3     set tx,copypc
           ld a,copypc
           ld pc,a
           nop
emet_4     ldi v,tbit1_e        ; tempo emission d'un bit
emet_5     dec v
           jrnz emet_5
           inc x                ; passe au bit suivant
           dec w
           jrz emet_7           ; si 8 bits transmis, emet stop bit
           jp emet_1
emet_7     nop                  ; 4 nop afin que le temps du huitime bit
           nop                  ; emis soit identique au 7 premiers
           nop
           nop
           set tx,copypc        ; force TX = 1 en conservant la configuration
           ld a,copypc          ; des entrees/sorties
           ld pc,a
           ldi v,tbit2_e
emet_6     dec v                ; tempo start bit
           jrnz emet_6
           ret

;--------------------------------------------------------------------
; routine de reception d'un caractere par liaison serie asynchrone
; le caractere recu est sauvegarde dans l'accu A et variable caracter
;--------------------------------------------------------------------

recoit     ldi drwr,maskbit0.w   ; pointe, via reg. X, la table des masques
           ldi x,maskbit0.d
recoit_0   ldi v,tbit0_r        ; reception bit de start ( tempo reg. V )
           ldi wdr,chien
recoit_1   jrs rx,pc,recoit_0   ; attend start bit
           ldi wdr,chien
           dec v                ; tempo start bit
           jrnz recoit_1
           ldi w,8              ; initialise le compteur de bits a 8
           ldi wdr,chien        ; relance le chien de garde
           clr a
           com a
           ld  caracter,a       ; initialise caratere recu
recoit_2   ldi v,tbit1_r
recoit_6   ldi wdr,chien
           dec v
           jrnz recoit_6
           ld a,caracter
           jrs rx,pc,recoit_3
           and a,(x)
           jp recoit_4
recoit_3   nop
           nop
recoit_4   ld caracter,a
           inc x
           dec w
           jrz recoit_5
           jp recoit_2
recoit_5   ldi v,tbit2_r
recoit_7   ldi wdr,chien
           dec v
           jrnz recoit_7
           ld a,caracter
           ret

;----------------------------------------------------------------------
; Routine de conversion binaire 16bits en dcimal cod ASCII 5 chiffres
;----------------------------------------------------------------------

ConvAscii   ldi w,5
            ldi y,ASCII4
            ldi drwr,TABP10.w
            ldi x,TABP10.d
            ld a,RegFreqT
            ld CCT0,a
            ld a,RegFreq0
            ld CCT1,a
TestLoop    ld a,(x)
            ld CPL,a
            inc x
            ld a,(x)
            ld CPH,a
            inc x
            ldi a,30h
            ld (y),a
TestSuite   ld a,CCT1
            cp a,CPH
            jrz Test0
            jrnc TestSup
            jp TestInf
Test0       ld a,CCT0
            cp a,CPL
            jrnc TestSup
            jp TestInf
TestSup     clr v
            ld a,CCT0
            sub a,CPL
            ld CCT0,a
            jrnc Test1
            ldi v,0FFh
Test1       ld a,CCT1
            sub a,CPH
            add a,v
            ld CCT1,a
            inc(y)
            ldi wdr,chien
            jp TestSuite
TestInf     dec y
            dec w
            jrz ConvFin
            jp TestLoop
ConvFin     ret


;---------------------------------------------
; Routine d'impression valeur ASCII 5 chiffres
;---------------------------------------------

AsciiOut    ldi y,ASCII4                    ; transmet les 5 chiffres
            ldi NBCAR,5                     ; contenus dans ASCII0  ASCII4
AscOut0     ld a,(y)                        ; vers la liaison srie
            call emet                       ; ASCII4=MSB et ASCII0=LSB
            dec y
            dec NBCAR
            jrnz AscOut0
            ret                             ; retour


;----------------------------------------------
; Routine de gestion de l'interruption du TIMER
;----------------------------------------------

gestim     res tmz,tscr         ; relance le timer ( le plus vite possible.)
           ldi tcr,ValTimer
           dec Temps1           ; 125 * 8 ms = 1s
           jrz tim_0            ; si seconde coule => mesure termine !
           reti                 ; sinon, continu la mesure
tim_0      res Enable,pb        ; inhibe comptage
           res 0,Status         ; indique fin de la mesure
           clr tscr             ; arrete les IT timer !
           reti

;----------------------------------------------
; Routine de gestion de l'interruption  de  PC4
;----------------------------------------------

gesPC4     jrs 0,Status,PC4_1   ; incremente seulement si mesure en cours...
           reti
PC4_1      inc RegFreq0
           jrz PC4_0
           reti
PC4_0      inc RegFreq1
           reti


;------------------------------------------
;  vecteurs d'interruption
;------------------------------------------

;  ***     CAN     ***
           .org 0FF0h
           reti

;  ***     TIMER   ***
           .ORG 0FF2h
           jp gestim

;  ***     PORTB,C ***
           .org 0FF4h
           jp gesPC4

;  ***     PORTA   ***
           .org 0FF6h
           reti

;  ***     NMI     ***
           .org 0FFCh
           reti

;  ***     RESET   ***
           .org 0FFEh
           jp init_pgm

           .end
