            .title "TEMPORISATEUR PROGRAMMABLE UNIVERSEL"
            .vers "st6225"
            .input "6225_reg.asm"
            .w_on


;----------------------------------------------------------------------------
;                          PRESENTATION DU PROGRAMME
;----------------------------------------------------------------------------
;
;
; ORGANISATION DU PROGRAMME
; =========================
;
; Des routines et macros ont t dveloppes afin d'afficher des informations
; sur un cran LCD dot d'un contrleur HD 44780 d'Hitachi
;
; La structure globale du programme montre les 6 parties importantes:
;
;         Dfinition des Variables en Ram
;         Dfinition des Constantes et Equivalences
;         Dfinition des Macros
;         CORPS DU PROGRAMME ( faisant appel aux macros et routines )
;         Dfinition des Routines
;         Dfinition des vecteurs d'interruption.
;
;

;-----------------------------------------------------------------------------
;                       Dfinition des constantes
;-----------------------------------------------------------------------------

chien       .equ 0FEh         ; valeur du chien de garde ( 24 mS environ )
busy        .equ 7            ; flag : busy=1 <=> HD44780 occup !
rs          .equ 6            ; slection instruction/donne ( HD 44780 )
rw          .equ 5            ; slection criture/lecture   ( HD 44780 )
e           .equ 4            ; validation criture/lecture  ( HD 44780 )
gauche      .equ 00011000b    ; dcalage affichage  gauche
droite      .equ 00011100b    ; dcalage affichage  droite
timilli     .equ 250          ; permet d'obtenir une IT toutes les 8 ms.
tmz         .equ 7            ; mis a 1 quand Timer(TCR) passe a 0.

synchro     .equ 0            ; bit de synchro d'affichage ( dans RegEtat )
StartT      .equ 1            ; indicateur de Temporisation On/Off
Speed       .equ 2            ; Indicateur vitesse de rglage
Alarme      .equ 3            ; indicateur tempo. termine 

Reglage     .equ 0            ; PB0 = bouton de reglage
StartStop   .equ 1            ; PB1 = bouton de start / stop temporisateur
regup       .equ 2            ; PB2 = bouton reglage "+"
regdown     .equ 3            ; PB3 = bouton reglage "-"

LSB         .equ 1            ; directives d'assemblage utilises par
MSB         .equ 1            ; la macro PRINTASC
NoLSB       .equ 0
NoMSB       .equ 0
UPTO        .equ 1            ; directives d'assemblage utilises par
DOWNTO      .equ 0            ; la macro NEXT
GOTO        .equ 0            ; directives d'assemblage utilises par
GOSBR       .equ 1            ; les macros ONSET et ONRESET
INCR        .equ 1            ; directives d'assemblage utilises par
DECR        .equ 0            ; la macro INCDEC

ModeTimer   .equ 01011100b    ; div. par 16 avec interruption autorise.

;-----------------------------------------------------------------------------
;                Dfinition des variables en ram $84  $BF
;-----------------------------------------------------------------------------

BaseRam     .def 084h
ii          .def BaseRam        ; registre controle de boucle 8bits
jj          .def ii+1           ; registre controle de boucle 8bits
reg0        .def jj+1           ; registre usage gnral
reg1        .def reg0+1         ; registre usage gnral
adrmesl     .def reg1+1         ; pointeurs de messages en EPROM
adrmesh     .def adrmesl+1
sava        .def adrmesh+1      ; sauvegarde accu A pout l'IT timer
curseur     .def sava+1         ; position curseur du caractre suivant
copypc      .def curseur+1      ; image RAM du port C ( diffrents modes )
millisec    .def copypc+1       ; comptage millisecondes.
TimSec      .def millisec+1     ; Dcomptage secondes.
TimMin      .def TimSec+1       ; Dcomptage minutes.
TimHeure    .def TimMin+1       ; Dcomptage heures.
RegEtat     .def TimHeure+1     ; Registre d'tat du temporisateur.
valconv     .def RegEtat+1      ; Sauvegarde de la conversion A->N
copypb      .def valconv+1      ; image RAM du port B
NBoucle1    .def copypb+1       ; nombre de bip sonores
NBoucle2    .def NBoucle1+1     ; nombre de salves sonores

;----------------------------------------------------------------------------
;             Dfinition des macro-instructions diverses ( affichage,...)
;----------------------------------------------------------------------------

; Macro d'initialisation des fonctions du micro-contrleur :
;   - mode de fonctionnement des ports parallles A, B et C
;          - Port C : PC3  PC6 en sortie symtrique,
;          - Port C : PC7 en entre logique avec R rappel sans IT.
;          - Port A : tout en sortie symtrique.
;          - Port B : PB0  PB3,PB5 et PB6 en entre avec R de rappel sans IT.
;          - Port B : PB4 en sortie  collecteur ouvert.
;          - Port B : PB7 en sortie symtrique
;   - mode de fonctionnement du TIMER
;          - mode TImer avec I.T toutes les 8mS.
;            Quartz 6MHz, Prescaler = 16 , Dcompte timer = 250 ,
;                               12
;            => I.T au bout de ---- x 16 x 250 = 8ms.
;                              6MHz
;   - mode de fonctionnement du convertisseur Analogique -> Numrique
;          - Sans IT.
;   - Validation/Autorisation des interruptions
;   - Mode de fonctionnement de l'afficheur LCD

            .MACRO RESET
            ldi pcdir,01111111b  ; initialisation des ports parallles
            ldi pcopt,11111111b  ; 
            ldi copypc,10000000b ; 
            ldi pc,10000000b
            ldi pbdir,10010000b  ; 
            ldi pbopt,10000000b  ;
            ldi pb,00010000b
            ldi copypb,00010000b
            ldi padir,0ffh       ;
            ldi paopt,0ffh
            ldi pa,0
            ldi v,6              ; pose de 60 ms
            call tempo
            ldi a,00110000b      ; initialisation logicielle de l'afficheur
            call outinstr        ; LCD juste au cas o ....
            ldi v,6
            call tempo
            ldi a,00110000b
            call outinstr
            ldi v,6
            call tempo
            ldi a,00110000b
            call outinstr
            ldi v,6
            call tempo
            ldi a,00111000b      ; mode 2 lignes, fonte 5x7
            call ecrinstr
            ldi a,00001000b      ; cran teint
            call ecrinstr
            ldi a,00000110b      ; mode incrmentation curseur
            call ecrinstr
            ldi a,00000001b      ; efface cran + curseur position 00
            call ecrinstr
            clr curseur          ; raz rfrence curseur
            ldi a,00001100b      ; allume cran sans curseur apparent
            call ecrinstr
            clr TimSec           ; init. temporisation
            clr TimMin
            clr TimHeure
            clr RegEtat
            ldi adcr,00000000b   ; CAN non aliment, pas d'interruptions
            ldi tcr,timilli      ; init. valeur timer
            ldi tscr,ModeTimer
            ldi ior,10h          ; autorise interruptions masquables
            reti                 ; place le micro-contrleur en mode NORMAL
            .ENDM


; Macro d'effacement de l'cran, curseur en dbut de ligne

            .MACRO CLEARHOME
            ldi a,00000001b
            call ecrinstr
            clr curseur
            .ENDM

; Macro de retour curseur en dbut de ligne

            .MACRO HOME
            ldi a,00000010b
            call ecrinstr
            clr curseur
            .ENDM

; Macro d'initialisation de la position du curseur ( 0 =< position =< 15 )

            .MACRO TABULATION poscurs
            .ifc ge poscurs - 16
            .error "position du curseur entre 0 et 15"
            .endc
            ldi curseur,poscurs
            .ifc ge poscurs - 8
            ldi a,80h+40h+poscurs-8
            .else
            ldi a,80h+poscurs
            .endc
            call ecrinstr
            .ENDM


; Macro d'envoi d'un message sur l'afficheur LCD  partir de la position
; courante du curseur.

            .MACRO PRINT msg_000 , ?lp5 , ?lp6
            ldi adrmesh,lp5.w
            ldi adrmesl,lp5.d
            call printmes
            jp lp6+1
lp5         .ascii msg_000
lp6         .byte 00
            .ENDM

; Macro d'envoi d'un message sur l'afficheur LCD.
; L'cran est effac et le curseur plac en dbut de ligne avant l'envoi
; du message.

            .MACRO PRINTCR msg_001 , ?lp7 , ?lp8
            ldi a,00000001b
            call ecrinstr
            clr curseur
            ldi adrmesh,lp7.w
            ldi adrmesl,lp7.d
            call printmes
            jp lp8+1
lp7         .ascii msg_001
lp8         .byte 00
            .ENDM


; Macro de conversion HEXA->ASCII du contenu d'une variable et envoi du
; rsultat sur l'afficheur LCD  la position courante du curseur.
;     cc1 = MSB  --> conversion et affichage des 4 bits MSB.
;     cc2 = LSB  --> conversion et affichage des 4 bits LSB.
;     var   = nom de la variable  convertir et afficher.
; utilisation : PRINTASC NomVaraible , NoMSB ou MSB , NoLSB ou LSB

            .MACRO PRINTASC  var , cc1 , cc2
            .ifc eq cc1 - 1
            ld a,var
            call HexAscH
            call ecridata
            .endc
            .ifc eq cc2 - 1
            ld a,var
            call HexAscL
            call ecridata
            .endc
            .ENDM

; Macro de conversion HEXA->BCD sur 8 bits ( avec N<100 ) du contenu d'une
; variable :
;     source   = nom de la variable  convertir et afficher.
;     dest1    = nom de la variable recevant la conversion au format BCD
; Rem : la conversion n'est effective que pour [source] < 100.

            .MACRO CVBCD100  source, dest1 , ?ll01 , ?ll02
            clr dest1
            ld a,source
ll01        cpi a,10
            jrc ll02
            inc dest1
            subi a,10
            jp ll01
ll02        ld v,a
            ld a,dest1
            sla a
            sla a
            sla a
            sla a
            add a,v
            ld dest1,a
            .ENDM



; Macro de temporisation, l'argument est en millisecondes ( de 10ms  2550mS )

            .MACRO POSE pp1
            ldi v,pp1/10
            call tempo
            .ENDM

; Macro de dcalage  gauche de la fentre vido de l'cran LCD.
; argument : Type  = gauche ou droite
; insr dans une boucle FOR ... NEXT permet de faire dfiler un texte.

            .MACRO DECALAGE type
            ldi a,type
            call ecrinstr
            .ENDM

; Macro d'inhibition de l'affichage LCD

            .MACRO DISPOFF
            ldi a,00001000b
            call ecrinstr
            .ENDM

; Macro de validation de l'affichage LCD

            .MACRO DISPON
            ldi a,00001100b
            call ecrinstr
            .ENDM

; Macro d'itration de type FOR ... NEXT
; Cette macro, utilise conjointement avec la macro NEXT, permet de programmer
; un nombre fini d'itration dans une structure logicielle de type
;    FOR i := NN DOWNTO valeur ...... NEXT i
; ou FOR i := NN UPTO   valeur ...... NEXT i
;
; utilisation :  FOR variable , valeur initiale , label
;                ....
;                NEXT variable , UPTO , valeur finale , label
;         ou     NEXT variable , DOWNTOTO , valeur finale , label
; Rem : la variable est de type Byte ( 8 bits seulement )


            .MACRO FOR  var1 , val1 , loopfor
            ldi var1,val1
loopfor     .equ $
            .ENDM


            .MACRO NEXT var2 , updown , val2 , saut2
            .ifc ge updown - 2
            .error " exclusivement: UPTO ou DOWNTO"
            .endc
            .ifc eq updown - UPTO
            inc var2
            .else
            dec var2
            .endc
            ld a,var2
            cpi a,val2
            jrz $+3
            jp saut2
            .ENDM


; Macro de saut  une adresse ( permet de rester cohrent avec la dmarche
; adopte qui consiste  rendre le corps du programme lisible et d'criture
; la plus proche possible d'un langage volu )

            .MACRO GOTO saut4
            jp saut4
            .ENDM

; Macro de saut  un sous programme ( permet de rester cohrent avec la
; dmarche adopte qui consiste  rendre le corps du programme lisible et
; d'criture la plus proche possible d'un langage volu )

            .MACRO GOSUB sbrsaut
            call sbrsaut
            .ENDM


; Macro de saut conditionn par l'tat "1" d'un bit d'une variable

            .MACRO ONSET bit , var , gowhere , labelset , ?l100
            .ifc ge gowhere - 2
            .error "ONSET utilise GOTO ou GOSBR"
            .endc
l100        jrr bit,var, l100+5
            .ifc eq gowhere - GOTO
            jp labelset
            .else
            call labelset
            .endc
            .ENDM

; Macro de saut conditionn par l'tat "0" d'un bit d'une variable

            .MACRO ONRESET bit , var , reswhere , labelres ,?l100
            .ifc ge reswhere - 2
            .error "ONRESET utilise GOTO ou GOSBR"
            .endc
l100        jrs bit,var,l100+5
            .ifc eq reswhere - GOTO
            jp labelres
            .else
            call labelres
            .endc
            .ENDM

; Macro d'initialisation d'une variable

            .MACRO SETVAR var , val
            ldi var,val
            .ENDM

            .MACRO INCDEC type , var , max , ?lpok1 , ?lpok2
            .ifc eq type-INCR
            ld a,var
            cpi a,max
            jrc lpok1
            ldi var,255
lpok1       inc var
            .endc
            .ifc eq type-DECR
            ld a,var
            jrnz lpok2
            ldi var,max+1
lpok2       dec var
            .endc
            .ENDM

; Macro de transfert d'une variable dans une autre

            .MACRO MOVE r1 ,r2
            ld a,r1
            ld r2,a
            .ENDM


;-----------------------------------------------------------------------------
;[[[[[[[[[[[[[[[[[[[[[[[[[  *  PROGRAMME PRINCIPAL *  ]]]]]]]]]]]]]]]]]]]]]]]]
;-----------------------------------------------------------------------------

           .org 080h


StartProg  ldi wdr,chien                    ; lance un os au chien de garde

           RESET                            ; initialise tout !
Prog       GOSUB AffTime0
Prog0      ONRESET Reglage  , pb , GOTO , SPreglage
           ONSET StartStop, pb , GOTO , Prog0
           ld a,TimHeure                    ; teste si temps valide ( <> 0 )
           jrnz NonZero
           ld a,TimMin
           jrnz NonZero
           ld a,TimSec
           jrnz NonZero
           GOTO Prog0                       ; non, attend nouveau rglage.
NonZero    POSE 100                         ; attend relachement de la touche
           ONRESET StartStop , pb , GOTO , NonZero
           ldi millisec,125                 ; oui, lance temporisation
           ld a,copypb                      ; active sortie PB4
           andi a,11101111b
           ld copypb,a
           ld pb,a
           set StartT,RegEtat               
           res Alarme,RegEtat               ; pas d'alarme pour le moment
SPstart    GOSUB AffTime1
heure      ONSET synchro , RegEtat , GOSBR , AffT0
           ONRESET StartT , RegEtat , GOTO , StopTime
           ONSET StartStop , pb , GOTO , heure
ToucheW    POSE 100                         ; attend relachement de la touche
           ONRESET StartStop , pb , GOTO , ToucheW
           clr RegEtat                      ; fin tempo. force...
           GOTO heure
StopTime   PRINTCR <"----- FIN ------">
           ld a,copypb                      ; dsactive sortie PB4
           andi a,11101111b
           addi a,00010000b
           ld copypb,a
           ld pb,a
           ONRESET Alarme , RegEtat , GOTO , Prog
           ; gnration de l'alarme sonore si ALARME = 1
           clr tscr                         ; dsactive Timer
           ldi NBoucle2,6
Son_S      ldi NBoucle1,5
Son_2      ldi v,50
Son_1      ld a,copypb                      ; active sortie PB7 (HP)
           andi a,01111111b
           addi a,10000000b
           ld copypb,a
           ld pb,a
           call TempsC                      ; delais de 1ms
           ld a,copypb                      ; dsactive sortie PB7
           andi a,01111111b
           ld copypb,a
           ld pb,a
           call TempsC                      ; delais de 1ms
           dec v
           jrz Son_0
           jp Son_1
Son_0      POSE 100
           dec NBoucle1
           jrz Son_3
           JP Son_2
Son_3      dec NBoucle2
           jrz Son_Fin
           POSE 500
           jp Son_S
Son_Fin    ldi tcr,timilli                  ; relance Timer
           ldi tscr,ModeTimer     
           GOTO Prog                        ; on recommence !

SPreglage  POSE 200
           ONSET Reglage , pb , GOTO , Prog

Attend0    POSE 200
           ONRESET Reglage , pb , GOTO , Attend0
           PRINTCR <"  00h  00m  00s ">
           CVBCD100 TimHeure , reg0
           TABULATION 2
           PRINTASC reg0 , MSB , LSB
           CVBCD100 TimMin , reg0
           TABULATION 7
           PRINTASC reg0 , MSB , LSB
           CVBCD100 TimSec , reg0
           TABULATION 12
           PRINTASC reg0 , MSB , LSB

RegHeure   TABULATION 1
           PRINT <"[">
           TABULATION 4
           PRINT <"]">

RegH0      RES Speed,RegEtat
RegH1      ONRESET regup , pb , GOTO , RegH3
           ONSET regdown , pb , GOTO , RegH2
           INCDEC DECR , TimHeure , 23
           GOTO RegH4
RegH3      INCDEC INCR , TimHeure , 23
RegH4      TABULATION 2
           CVBCD100 TimHeure , reg0
           PRINTASC reg0 , MSB , LSB
           POSE 200
           ONRESET regdown , pb , GOTO , RegH5
           ONSET regup , pb , GOTO , RegH2
RegH5      ONSET Speed , RegEtat , GOTO , RegH1
           POSE 500
           SET Speed,RegEtat
           GOTO RegH1
RegH2      ONRESET StartStop , pb , GOTO , RegOk
           ONSET Reglage , pb , GOTO , RegH0

           TABULATION 1
           PRINT <" ">
           TABULATION 4
           PRINT <"h">

Attend1    POSE 200
           ONRESET Reglage , pb , GOTO , Attend1

RegMin     TABULATION 6
           PRINT <"[">
           TABULATION 9
           PRINT <"]">

RegM0      RES Speed,RegEtat
RegM1      ONRESET regup , pb , GOTO , RegM3
           ONSET regdown , pb , GOTO , RegM2
           INCDEC DECR , TimMin , 59
           GOTO RegM4
RegM3      INCDEC INCR , TimMin , 59
RegM4      TABULATION 7
           CVBCD100 TimMin , reg0
           PRINTASC reg0 , MSB , LSB
           POSE 200
           ONRESET regdown , pb , GOTO , RegM5
           ONSET regup , pb , GOTO , RegM2
RegM5      ONSET Speed , RegEtat , GOTO , RegM1
           POSE 500
           SET Speed,RegEtat
           GOTO RegM1
RegM2      ONRESET StartStop , pb , GOTO , RegOk
           ONSET Reglage , pb , GOTO , RegM0

           TABULATION 6
           PRINT <" ">
           TABULATION 9
           PRINT <"m">

Attend2    POSE 200
           ONRESET Reglage , pb , GOTO , Attend2

RegSec     TABULATION 11
           PRINT <"[">
           TABULATION 14
           PRINT <"]">

RegS0      RES Speed,RegEtat
RegS1      ONRESET regup , pb , GOTO , RegS3
           ONSET regdown , pb , GOTO , RegS2
           INCDEC DECR , TimSec , 59
           GOTO RegS4
RegS3      INCDEC INCR , TimSec , 59
RegS4      TABULATION 12
           CVBCD100 TimSec , reg0
           PRINTASC reg0 , MSB , LSB
           POSE 200
           ONRESET regdown , pb , GOTO , RegS5
           ONSET regup , pb , GOTO , RegS2
RegS5      ONSET Speed , RegEtat , GOTO , RegS1
           POSE 500
           SET Speed,RegEtat
           GOTO RegS1
RegS2      ONRESET StartStop , pb , GOTO , RegOk
           ONSET Reglage , pb , GOTO , RegS0

           TABULATION 11
           PRINT <" ">
           TABULATION 14
           PRINT <"s">

Attend3    POSE 200
           ONRESET Reglage , pb , GOTO , Attend3
           GOTO RegHeure

RegOk      POSE 200
           ONRESET StartStop , pb , GOTO , RegOk
           GOTO Prog

          

AffTime0   PRINTCR <"Temps:   h  m  s">
           GOTO AffT0
AffTime1   PRINTCR <"Reste:   h  m  s">
AffT0      CVBCD100 TimHeure , reg0
           TABULATION 7
           PRINTASC reg0 , MSB , LSB
           CVBCD100 TimMin , reg0
           TABULATION 10
           PRINTASC reg0 , MSB , LSB
           CVBCD100 TimSec , reg0
           TABULATION 13
           PRINTASC reg0 , MSB , LSB
           res synchro,RegEtat
           ret



;-----------------------------------------------------------------------------
;[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]
;-----------------------------------------------------------------------------




;----------------------------------------------------------------------------
;             Dfinition des routines diverses ( affichage,...)
;----------------------------------------------------------------------------

; Gestion de l'interruption du TIMER ( appele toutes les 8 ms ).

IntTimer   res tmz,tscr         ; pour la prochaine interruption
           ldi tcr,timilli      ; recharge timer et le chien de garde.
           ldi wdr,chien
           ld sava,a            ; sauvegarde l'accu A
           jrs StartT,RegEtat,tim_on
           jp tim_fin

tim_on     dec millisec         ; Une seconde coule ?
           jrz tim_0            ; oui, va incrmenter les secondes
           jp tim_fin           ; non, rentre  la maison.
tim_0      ldi millisec,125     ; 125 * 8ms = 1s
           set synchro,RegEtat  ; Indique une seconde coule
           ld a,TimSec
           addi a,0FFh          ; dcrmente les secondes
           jrnc tim_1
           ld TimSec,a          
           jp tim_test
tim_1      ldi TimSec,59        ; dcrmente les minutes
           ld a,TimMin
           addi a,0FFh          
           jrnc tim_2
           ld TimMin,a          ; dcrmente les secondes
           jp tim_test
tim_2      ldi TimMin,59        ; dcrmente les heures
           dec TimHeure
tim_test   ld a,TimSec
           add a,TimMin
           add a,TimHeure
           jrnz tim_fin
           res StartT,RegEtat   ; fin Temporisation
           set Alarme,RegEtat   ; Pour alarme sonore
tim_fin    ld a,sava            ; restaure l'accu A
           reti

; Temporisation logicielle programmable de 10ms  256*10ms # 2,56S par bonds
; de 10 ms. Le nombre de 10 millisecondes doit tre charg dans V avant
; l'appel de la routine "tempo". ( valable pour Quartz 6MHz )

tempo       ldi w,209           ;   209 pour quartz 6MHz
temp0       ldi wdr,chien
            nop
            nop
            nop
            dec w
            jrnz temp0
            dec v
            jrnz tempo
            ret

; Temporisation logicielle de 1ms (utilise pour l'alarme sonore)

TempsC      ldi w,21           ;   21 pour quartz 6MHz
TempsC0     ldi wdr,chien
            nop
            nop
            nop
            dec w
            jrnz TempsC0
            ret

; Test du bit d'tat BUSY de l'afficheur LCD
; L'tat de l'indicateur BUSY est renvoy dans la CARRY

testbusy    ldi wdr,chien
            clr padir            ; port A en entre car on effectue une lecture
            clr paopt
            ldi pa,255           ; sans rsistances de rappel ni interruption

            ld a,copypc          ; travaille sur image du port C
            res rs,a             ; impose RS = 0 car c'est une instruction
            res e,a              ; impose  E = 0 pour le moment
            set rw,a             ; impose RW = 1 ( lecture )
            ld copypc,a
            ld pc,a

            ld a,copypc
            set e,a              ; impose  E = 1 ( validation de la lecture )
            ld copypc,a
            ld pc,a              ; c'est envoy ...
            nop
            nop

            ld a,pa              ; lit l'tat du processeur graphique
            ld v,a               ; et le sauvegarde

            ld a, copypc         ; inhibe la lecture en cours
            res e,a
            ld copypc,a
            ld pc,a

            ld a, copypc         ; annule l'action de lecture en cours
            res rw,a
            ld copypc,a
            ld pc,a

            ldi padir,255        ; reconfigure port A en sortie symtrique
            ldi paopt,255
            clr pa
            ldi wdr,chien
            jrs busy,v,fintest   ; transfert : CARRY = BUSY
fintest     ret



; Routine d'criture d'une instruction  destination de l'afficheur LCD
; L'instruction  crire est contenu dans l'accu A
; ( initialement E = RW = RS = 0 , c'est l'tat par dfaut ! )
; REM : l'entre en OUTINSTR permet d'crire une instruction sans tenir
;       compte de l'tat de BUSY ( c'est le cas lors d'une initilisation
;       logicielle ).

ecrinstr    ld w,a               ; sauvegarde l'instruction  crire
ecr_0       call testbusy        ; test si le 44780 travaille
            jrc ecr_0            ; oui, on attend
            ld a,w               ; rcupre le code de l'instruction
outinstr    ld pa,a              ; non, impose l'instruction  crire
            ld a,copypc          ; travaille sur image du port C
            set e,a              ; impose  E = 1 pour valider l'criture
            ld copypc,a
            ld pc,a
            nop
            nop
            ld a,copypc          ; travaille sur image du port C
            res e,a              ; impose  E = 0 pour valider l'criture
            ld copypc,a          ; ( l'criture se fait sur front descendant )
            ld pc,a
            nop
            nop
            clr pa               ; annule l'instruction ( histoire de ...)
            ldi wdr,chien
            ret



; Routine d'criture d'une donne  destination de l'afficheur LCD
; La donne  crire est contenu dans l'accu A
; ( initialement E = RW = RS = 0 , c'est l'tat par dfaut ! )

ecridata    ldi wdr,chien
            ld w,a               ; sauvegarde la donne  crire
ecr_4       call testbusy
            jrc ecr_4
            ld a,w               ; rcupre la donne
            ld pa,a              ; impose la donne  crire
            ld a,copypc          ; travaille sur image du port C
            set rs,a             ; impose RS = 1 ( tranfert de donnes )
            ld copypc,a
            ld pc,a
            nop
            nop
            ld a,copypc          ; travaille sur image du port C
            set e,a              ; impose  E = 1 pour valider l'criture
            ld copypc,a
            ld pc,a
            nop
            nop
            ld a,copypc          ; travaille sur image du port C
            res e,a              ; impose  E = 0 pour valider l'criture
            ld copypc,a          ; ( l'criture se fait sur front descendant )
            ld pc,a
            nop
            nop
            ld a,copypc          ; travaille sur image du port C
            res rs,a             ; impose  RS = 0
            ld copypc,a
            ld pc,a
            nop
            nop
            inc curseur          ; incrmente position curseur
            ld a,curseur         ; non, teste si position curseur >= 8
            cpi a,8
            jrnz ecr_5
            ldi a,80h+40h        ; oui, saut  la 2ime partie de l'cran
            jp ecrinstr          ; le RET se fera en fin de routine ECRINSTR
ecr_5       clr pa               ; raz port A, juste pour le fun ...
            ret

; routine d'mission d'une chaine de caractres en zone ROM
; ADRMESH  doit contenir les 6 bits de poids forts de l'adresse du message
; ADRMESL  doit contenir les 6 bits de poids faibles de l'adresse du message
; Le message doit se terminer par le caractre nul 00h
; Les registres utiliss sont : A , X
; L'appel est de la forme :  LDI ADRMESH, @.W
;                            LDI ADRMESL, @.D
;                            CALL PRINTMES

printmes    ld a,adrmesl
            ld x,a
pmes_1      ld a,adrmesh         ; recupre adresse du caractere courant
            ld drwr,a
pmes_2      ld a,(x)             ; recupre caractre courant du message
            jrnz pmes_0          ; si fin de message on rentre  la maison !
            ret
pmes_0      call ecridata        ; envoit le caractre sur afficheur LCD
            inc x                ; passe au caractre suivant
            jrr 7,x,pmes_2       ; teste si il faut incrmenter le pointeur de
            ldi x,40h            ; la fentre en ROM.
            inc adrmesh          ; Si oui, initialise  6 bits de poids failes
            jp pmes_1            ; et incrmente les 6 bits de poids forts.



;-------------------------------------------
; Routine de conversion   hexa4 LSB -> ASCII
;-------------------------------------------

HexAscL    andi a,0Fh
           cpi a,10
           jrnc Hex_0
           addi a,'0'
           ret
Hex_0      subi a,10
           addi a,'A'
           ret

;-------------------------------------------
; Routine de conversion   hexa4 MSB -> ASCII
;-------------------------------------------

HexAscH    rlc a
           rlc a
           rlc a
           rlc a
           rlc a
           jp HexAscL


;----------------------------------------------------------------------------
;                        Vecteurs d'interruption
;----------------------------------------------------------------------------

;  ***     CAN     ***
           .org 0FF0h
           nop
           reti

;  ***     TIMER   ***
           .ORG 0FF2h
           jp IntTimer

;  ***     PORTB,C ***
           .org 0FF4h
           nop
           reti

;  ***     PORTA   ***
           .org 0FF6h
           nop
           reti

;  ***     NMI     ***
           .org 0FFCh
           nop
           reti

;  ***     RESET   ***
           .org 0FFEh
           jp StartProg

           .end
