           .title "Interrupteur Crepusculaire Programmable"
           .vers  "st6225"
           .input "6225_reg.asm"
           .w_on

; Mode de fonctionnement des ports parallles A, B et C
;          - Port A : tout en entre logique avec R de rappel sans IT
;          - Port B : PB0 et PB2 : sortie symtrique
;                     PB1        : sortie logique  drain ouvert
;                     PB3  PB7  : entre logique avec R de rappel sans IT
;          - Port C : PC4        : entre logique sans R de rappel ni IT.
;                     PC0  PC3  : entre logique avec R de rappel sans IT.
;
;  !---------!-------------------!-----------------!------------------!
;  ! ....... !  DDR  direction   !   OR   option   !  DR   donnees    !
;  !---------!-------------------!-----------------!------------------!
;  ! port A  !  00000000 =   00h ! 00000000 = 00h  ! 00000000 = 00h   !
;  !---------!-------------------!-----------------!------------------!
;  ! port B  !  00000111 =   07h ! 00000101 = 05h  ! 00000010 = 02h   !
;  !---------!-------------------!-----------------!------------------!
;  ! port C  !  00000000 =   00h ! 00000000 = 00h  ! 00010000 = 10h   !
;  !---------!-------------------!-----------------!------------------!
;
;
;-----------------------------------------------------------------------------
;                          Definition des constantes
;-----------------------------------------------------------------------------

chien      .equ 0FEh            ; valeur du chien de garde
tmz        .equ 7               ; mis a 1 quand Timer(TCR) passe a 0.
ValTimer   .equ 84              ; pour base de temps de 8ms
TimerMode  .equ 01011110b       ; Division par 64, IT timer autorises
                                ; ( 1.5s * 64 * 84 = 8.064ms ) 
Nuit       .equ 4               ; PC4 = indicateur de nuit (absence lumire)
LedNuit    .equ 1               ; PB1 = commande LED indicatrice de la nuit.
RunStop    .equ 0               ; bit0=1 de RegEtat <=> temporisation lance
TMnuit     .equ 15              ; temps mort dtection nuit (15min)
TMjour     .equ 60              ; temps mort dtection jour (60min)

;-----------------------------------------------------------------------------
;                  Definition des variables en ram $84  $BF
;-----------------------------------------------------------------------------


BaseRam    .def 084h            ; Dbut de la zone RAM
copypb     .def BaseRam         ; registre image du port B
MilliSec   .def copypb+1        ; comptage des millisecondes
Secondes   .def MilliSec+1      ; comptage des secondes
Tempor     .def Secondes+1      ; dcomptage des minutes
RegEtat    .def Tempor+1        ; registre d'tat du temporisateur
save_a     .def RegEtat+1       ; sauvegarde de l'accu. A pendant I.T. TIMER

;-----------------------------------------------------------------------------
;                             Debut du programme
;-----------------------------------------------------------------------------

           .org 080h

StartPGM   ldi wdr,chien          ; rarme le chien de garde
           clr padir              ; Initialisation mode de fonctionnement    
           clr paopt              ; des ports parallles
           clr pa
           ldi pbdir,07h          ; 
           ldi pbopt,05h
           ldi pb,02h 
           ldi copypb,02h
           ldi pcdir,00h          ;                                 
           ldi pcopt,00h          ;                              
           ldi pc,10h 
           ldi MilliSec,125       ; 125*8ms = 1 seconde
           ldi Secondes,60        ; 60*1s=1minute
           clr Tempor             ; tempo. nulle pour le moment
           clr RegEtat            ; pas de tempo en cours
           ldi tcr,ValTimer
           ldi tscr,TimerMode     ; mode de fonctionnement du TIMER
           ldi ior,00010000b      ; autorise interruptions masquables.
           reti                   ; revient au mode normal.

Cycle0     res RunStop,RegEtat
           ldi MilliSec,125       ; initialise tempo pralable
           ldi Secondes,60        ; Il doit faire sombre pendant un certain
           ldi Tempor,TMnuit      ; temps avant d'allumer les lampes
Attend_N   jrr Nuit,pc,Attend_N   ; on attend la nuit...
           set RunStop,RegEtat    ; La nuit est l, active temporisation
Attend_S   jrr Nuit,pc,Cycle0     ; dj de la lumire ? fausse alerte!
           jrs RunStop,RegEtat,Attend_S

           ; c'est bien la nuit qui tombe...

           res RunStop,RegEtat    ; pas de tempo pour le moment
           ldi MilliSec,125       ; initialise temporisation en fonction 
           ldi Secondes,60        ; de la dure programme sur la carte
           ld a,pa
           ld Tempor,a
           set 0,copypb           ; active relais
           set 2,copypb           ; active triac
           set RunStop,RegEtat    ; active temporisation d'allumage des lampes
FinN       jrs RunStop,RegEtat,FinN  ; attend la fin de la tempo.
           res 0,copypb           ; dsactive relais
           res 2,copypb           ; dsactive triac

           ; maintenant il faut attendre le jour avec la mme scurit

Cycle1     res RunStop,RegEtat
           ldi MilliSec,125       ; initialise tempo pralable
           ldi Secondes,60        ; Il doit faire jour pendant un certain
           ldi Tempor,TMjour      ; temps avant de relancer le cycle
Attend_0   jrs Nuit,pc,Attend_0   ; on attend le jour...
           set RunStop,RegEtat    ; La nuit est l, active temporisation
Attend_1   jrs Nuit,pc,Cycle1     ; de nouveau la nuit ? fausse alerte!
           jrs RunStop,RegEtat,Attend_1

           ; c'est bien le jour qui revient...

           jp Cycle0              ; va attendre la prochaine nuit!
           



;-----------------------------------------------------------------------------
;                       Routine d'interruption du TIMER
;-----------------------------------------------------------------------------

GesTimer   res tmz,tscr            ; relance le TIMER
           ldi tcr,ValTimer
           ldi wdr,chien
           ld save_a,a

           ; dtection en temps rel de l'tat nuit / jour

           jrr Nuit,pc,Pas_Nuit   ; fait-il nuit ?
           res LedNuit,copypb     ; oui, allume Led concerne.
           jp Gest_6
Pas_Nuit   set LedNuit,copypb     ; non, teint led concerne.

           ; gestion de la temporisation

Gest_6     jrs RunStop,RegEtat,Gest_1
           jp Gest_Fin
Gest_1     dec MilliSec           ; dcrmente les millisecondes
           jrz Gest_7
           jp Gest_Fin
Gest_7     ldi MilliSec,125       ; init. nouveau cycle de 1 seconde
           dec Secondes           ; dcrmente les secondes
           jrz Gest_4
           jp Gest_Fin
Gest_4     ldi Secondes,60        ; init. nouveau cycle de 1 minute
           dec Tempor             ; dcrmente les minutes
           jrnz Gest_Fin
           res RunStop,RegEtat    ; temporisation termine
Gest_Fin   ld a,copypb
           ld pb,a
           ld a,save_a            ; restitue le contenu de l'accu A
           reti     


;-----------------------------------------------------------------------------
;                         Vecteurs d'interruption
;-----------------------------------------------------------------------------

;  ***     CAN     ***
           .org 0FF0h
           reti

;  ***     TIMER   ***
           .ORG 0FF2h
           jp GesTimer

;  ***     PORTB,C ***
           .org 0FF4h
           reti

;  ***     PORTA   ***
           .org 0FF6h
           reti

;  ***     NMI     ***
           .org 0FFCh
           reti

;  ***     RESET   ***
           .org 0FFEh
           jp StartPGM

           .end
