            .title "VOLTMETRE DOUBLE"
            .vers "st6225"
            .input "6225_reg.asm"
            .w_on

;-----------------------------------------------------------------------------
;                       Dfinition des constantes
;-----------------------------------------------------------------------------

chien       .equ 0FEh         ; valeur du chien de garde ( 24 mS environ )
busy        .equ 7            ; flag : busy=1 <=> HD44780 occup !
rs          .equ 6            ; slection instruction/donne ( HD 44780 )
rw          .equ 5            ; slection criture/lecture   ( HD 44780 )
e           .equ 4            ; validation criture/lecture  ( HD 44780 )
Virgule     .equ '.'
Volts       .equ 'V'

;-----------------------------------------------------------------------------
;                Dfinition des variables en ram $84  $BF
;-----------------------------------------------------------------------------

BaseRam     .def 084h
adrmesl     .def BaseRam        ; pointeurs de messages en EPROM
adrmesh     .def adrmesl+1
curseur     .def adrmesh+1      ; position curseur du caractre suivant
ASCII0      .def curseur+1      ; utilis pour conversion ASCII sur 3 chiffres
ASCII1      .def ASCII0+1
ASCII2      .def ASCII1+1
copypc      .def ASCII2+1       ; image en RAM du port C
CAN1        .def copypc+1       ; valeur canal 1
CAN2        .def CAN1+1         ; valeur canal 2


;----------------------------------------------------------------------------
;             Dfinition des macro-instructions diverses ( affichage,...)
;----------------------------------------------------------------------------

; Macro d'initialisation des fonctions du micro-contrleur :
;   - Mode de fonctionnement des ports parallles A, B et C
;          - Port A : tout en sortie symtrique.
;          - Port B : PB0 et PB1 entre analogique / entre logique.
;            PB0 et PB1 : initialement en entre sans R de rappel ni I.T
;                         Conversion A/N sur PB0: PB0 en entre analogique
;                         Conversion A/N sur PB1: PB1 en entre analogique
;          - Port B : PB2  PB7 en entre logique avec R de rappel sans IT.
;          - Port C : tout en sortie symtrique.
;
;  !---------!-------------------!-----------------!------------------!
;  ! ....... !  DDR  direction   !   OR   option   !  DR   donnees    !
;  !---------!-------------------!-----------------!------------------!
;  ! port A  !  11111111 =   FFh ! 11111111 = FFh  ! 00000000 = 00h   !
;  !---------!-------------------!-----------------!------------------!
;  ! port B  !  00000000 =   00h ! 00000000 = 00h  ! 00000011 = 03h   !
;  !---------!-------------------!-----------------!------------------!
;  ! port C  !  11110000 =   F0h ! 11110000 = F0h  ! 00000000 = 00h   !
;  !---------!-------------------!-----------------!------------------!

;   - Mode de fonctionnement du TIMER : non utilis.
;   - Mode de fonctionnement du convertisseur A/N : Sans IT.
;   - Mode de fonctionnement de l'afficheur LCD

            .MACRO RESET
            ldi pcdir,11110000b  ; initialisation des ports parallles
            ldi pcopt,11110000b  ; 
            ldi copypc,00000000b ; 
            ldi pc,00000000b
            ldi pbdir,00000000b  ; 
            ldi pbopt,00000000b  ;
            ldi pb,00000011b
            ldi padir,0FFh       ;
            ldi paopt,0FFh
            ldi pa,0
            ldi v,6              ; pose de 60 ms
            call tempo
            ldi a,00110000b      ; initialisation logicielle de l'afficheur
            call outinstr        ; LCD juste au cas o ....
            ldi v,6
            call tempo
            ldi a,00110000b
            call outinstr
            ldi v,6
            call tempo
            ldi a,00110000b
            call outinstr
            ldi v,6
            call tempo
            ldi a,00111000b      ; mode 2 lignes, fonte 5x7
            call ecrinstr
            ldi a,00001000b      ; cran teint
            call ecrinstr
            ldi a,00000110b      ; mode incrmentation curseur
            call ecrinstr
            ldi a,00000001b      ; efface cran + curseur position 00
            call ecrinstr
            clr curseur          ; raz rfrence curseur
            ldi a,00001100b      ; allume cran sans curseur apparent
            call ecrinstr
            ldi adcr,90h         ; CAN actif avec interruptions
            ldi tscr,00          ; timer inutilis
            ldi ior,10h          ; autorise interruptions masquables
                                 ; reste en mode NMI.
            .ENDM


; Macro d'effacement de l'cran, curseur en dbut de ligne

            .MACRO CLEARHOME
            ldi a,00000001b
            call ecrinstr
            clr curseur
            .ENDM

; Macro de retour curseur en dbut de ligne

            .MACRO HOME
            ldi a,00000010b
            call ecrinstr
            clr curseur
            .ENDM

; Macro d'initialisation de la position du curseur ( 0 =< position =< 15 )

            .MACRO TABULATION poscurs
            .ifc ge poscurs - 16
            .error "position du curseur entre 0 et 15"
            .endc
            ldi curseur,poscurs
            .ifc ge poscurs - 8
            ldi a,80h+40h+poscurs-8
            .else
            ldi a,80h+poscurs
            .endc
            call ecrinstr
            .ENDM


; Macro d'envoi d'un message sur l'afficheur LCD  partir de la position
; courante du curseur.

            .MACRO PRINT msg_000 , ?lp5 , ?lp6
            ldi adrmesh,lp5.w
            ldi adrmesl,lp5.d
            call printmes
            jp lp6+1
lp5         .ascii msg_000
lp6         .byte 00
            .ENDM

; Macro d'envoi d'un message sur l'afficheur LCD.
; L'cran est effac et le curseur plac en dbut de ligne avant l'envoi
; du message.

            .MACRO PRINTCR msg_001 , ?lp7 , ?lp8
            ldi a,00000001b
            call ecrinstr
            clr curseur
            ldi adrmesh,lp7.w
            ldi adrmesl,lp7.d
            call printmes
            jp lp8+1
lp7         .ascii msg_001
lp8         .byte 00
            .ENDM


; Macro d'affichage d'une valeur ASCII  3 chiffres en tenant compte de la
; position de la virgule.
; [Source]=centaines , [Source-1]=dizaines , [Source-2]=units
;      Si Canal   = 1   =>     teste PB7 pour position virgule canal 1
;      Si Canal   = 2   =>     teste PB6 pour position virgule canal 2

            .MACRO PRINTASC  source , canal , ?lpp01 , ?lpp02 , ?lpp03

            ld a,source             ; affiche les centaines
            call ecridata

            .ifc eq canal - 1       ; teste position virgule canal 1
            ld a,pb
            andi a,80h
            .endc

            .ifc eq canal - 2       ; teste position virgule canal 2
            ld a,pb
            andi a,40h
            .endc

            jrnz lpp01              ; Z=0 si calibre 2,5V
            jp lpp02                ; Z=1 si calibre 25V
lpp01       ldi a,Virgule           ; affiche au format X.XX
            call ecridata
            ld a,source-1
            call ecridata
            ld a,source-2
            call ecridata
            jp lpp03
lpp02       ld a,source-1           ; affiche au format XX.X    
            call ecridata
            ldi a,Virgule
            call ecridata
            ld a,source-2
            call ecridata
lpp03       ldi a,Volts             ; la valeur est suivie de "V"
            call ecridata           ; Ex: 0.56V
            .ENDM


; Macro de conversion HEXA->ASCII sur 8 bits du contenu d'une variable :
;     source   = nom de la variable  convertir et afficher.
;     dest1    = nom de la variable recevant le MSB converti ASCII
; Rem : aprs la conversion [dest1]  = centaines au format ASCII ,
;                           [dest1-1]= dizaines au format ASCII ,
;                           [dest1-2]= units au format ASCII .
;
; Exemple : si [source]=0A5h <=> 165 , alors  [dest1]  = 31h  ('1')
;                                             [dest1-1]= 36h  ('6')
;                                             [dest1-2]= 35h  ('5')
; Cette macro utilise le registre A

            .MACRO CONVASCII  source, dest1 , ?ll01 , ?ll02 , ?ll03
            ldi a,30h
            ld dest1,a
            ld dest1-1,a
            ld dest1-2,a
            ld a,source
ll01        cpi a,100
            jrc ll02
            inc dest1
            subi a,100
            jp ll01
ll02        cpi a,10
            jrc ll03
            inc dest1-1
            subi a,10
            jp ll02
ll03        add a,dest1-2
            ld dest1-2,a
            .ENDM



; Macro de temporisation, l'argument est en millisecondes ( de 10ms  2550mS )

            .MACRO POSE pp1
            ldi v,pp1/10
            call tempo
            .ENDM

; Macro de dcalage  gauche de la fentre vido de l'cran LCD.
; argument : Type  = gauche ou droite
; insr dans une boucle FOR ... NEXT permet de faire dfiler un texte.

            .MACRO DECALAGE type
            ldi a,type
            call ecrinstr
            .ENDM

; Macro d'inhibition de l'affichage LCD

            .MACRO DISPOFF
            ldi a,00001000b
            call ecrinstr
            .ENDM

; Macro de validation de l'affichage LCD

            .MACRO DISPON
            ldi a,00001100b
            call ecrinstr
            .ENDM

; Macro de conversion analogique / numrique
; canal = 1 : PB0 en entre analogique
; canal = 2 : PB1 en entre analogique
; Var   = variable de destination de la conversion

            .MACRO GETANALOG canal , var , ?lpcv

            .ifc eq canal - 1     ; si canal = 1
            set 0,pbopt           ; PB0 en entre analogique
            .endc

            .ifc eq canal - 2     ; si canal = 2
            set 1,pbopt           ; PB1 en entre analogique
            .endc
            nop
            nop                   ; petit dlai pour stabilit de l'entre
            set 5,adcr            ; lance conversion
            wait                  ; mode wait pour minimiser le bruit...
lpcv        ldi wdr,chien         ; attend fin de conversion
            jrr 6,adcr,lpcv
            res 5,adcr            ; CAN prt pour la prochaine conversion
            ld a,adr              ; recupre la valeur convertie

            .ifc eq canal - 1     ; si canal = 1
            res 0,pbopt           ; PB0 en entre logique sans rappel ni IT
            .endc

            .ifc eq canal - 2     ; si canal = 2
            res 1,pbopt           ; PB1 en entre logique sans rappel ni IT
            .endc
        
            ld var,a              ; mmorise la valeur convertie
            .ENDM

;-----------------------------------------------------------------------------
;[[[[[[[[[[[[[[[[[[[[[[[[[  *  PROGRAMME PRINCIPAL *  ]]]]]]]]]]]]]]]]]]]]]]]]
;-----------------------------------------------------------------------------

           .org 080h


StartProg  ldi wdr,chien                    ; lance un os au chien de garde
           RESET                            ; initialise tout !

Mesure     GETANALOG 1 , CAN1               ; conversion canal 1
           GETANALOG 2 , CAN2               ; conversion canal 2
           HOME                             ; retour curseur afficheur
           ld a,CAN1                        ; tension < ou =  2,50V ?
           cpi a,251
           jrnc Aff_S0                      ; non, affiche "-----"
           jp Aff_S00                       ; oui, affiche valeur tension
Aff_S0     PRINT <"-----">                   
           jp Aff_S1
Aff_S00    CONVASCII CAN1 , ASCII2          ; conversion ASCII mesure canal 1
           PRINTASC ASCII2 , 1              ; affichage valeur canal 1
Aff_S1     PRINT <"  :  ">
           ld a,CAN2                        ; tension < ou =  2,50V ?
           cpi a,251
           jrnc Aff_S2                      ; non, affiche "-----"
           jp Aff_S20                       ; oui, affiche valeur tension
Aff_S2     PRINT <"-----">                   
           jp Aff_S3
Aff_S20    CONVASCII CAN2 , ASCII2          ; conversion ASCII mesure canal 2
           PRINTASC ASCII2 , 2              ; affichage valeur canal 2
Aff_S3     POSE 500                         ; delais 0,5S
           jp Mesure



;-----------------------------------------------------------------------------
;[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[[]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]]
;-----------------------------------------------------------------------------




;----------------------------------------------------------------------------
;             Dfinition des routines diverses ( affichage,...)
;----------------------------------------------------------------------------


; Temporisation logicielle programmable de 10ms  256*10ms # 2,56S par bonds
; de 10 ms. Le nombre de 10 millisecondes doit tre charg dans V avant
; l'appel de la routine "tempo".

tempo       ldi w,255           
temp0       ldi wdr,chien
            nop
            nop
            nop
            dec w
            jrnz temp0
            dec v
            jrnz tempo
            ret


; Test du bit d'tat BUSY de l'afficheur LCD
; L'tat de l'indicateur BUSY est renvoy dans la CARRY

testbusy    ldi wdr,chien
            clr padir            ; port A en entre car on effectue une lecture
            clr paopt
            ldi pa,255           ; sans rsistances de rappel ni interruption

            ld a,copypc          ; travaille sur image du port C
            res rs,a             ; impose RS = 0 car c'est une instruction
            res e,a              ; impose  E = 0 pour le moment
            set rw,a             ; impose RW = 1 ( lecture )
            ld copypc,a
            ld pc,a

            ld a,copypc
            set e,a              ; impose  E = 1 ( validation de la lecture )
            ld copypc,a
            ld pc,a              ; c'est envoy ...
            nop
            nop

            ld a,pa              ; lit l'tat du processeur graphique
            ld v,a               ; et le sauvegarde

            ld a, copypc         ; inhibe la lecture en cours
            res e,a
            ld copypc,a
            ld pc,a

            ld a, copypc         ; annule l'action de lecture en cours
            res rw,a
            ld copypc,a
            ld pc,a

            ldi padir,255        ; reconfigure port A en sortie symtrique
            ldi paopt,255
            clr pa
            ldi wdr,chien
            jrs busy,v,fintest   ; transfert : CARRY = BUSY
fintest     ret



; Routine d'criture d'une instruction  destination de l'afficheur LCD
; L'instruction  crire est contenu dans l'accu A
; ( initialement E = RW = RS = 0 , c'est l'tat par dfaut ! )
; REM : l'entre en OUTINSTR permet d'crire une instruction sans tenir
;       compte de l'tat de BUSY ( c'est le cas lors d'une initilisation
;       logicielle ).

ecrinstr    ld w,a               ; sauvegarde l'instruction  crire
ecr_0       call testbusy        ; test si le 44780 travaille
            jrc ecr_0            ; oui, on attend
            ld a,w               ; rcupre le code de l'instruction
outinstr    ld pa,a              ; non, impose l'instruction  crire
            ld a,copypc          ; travaille sur image du port C
            set e,a              ; impose  E = 1 pour valider l'criture
            ld copypc,a
            ld pc,a
            nop
            nop
            ld a,copypc          ; travaille sur image du port C
            res e,a              ; impose  E = 0 pour valider l'criture
            ld copypc,a          ; ( l'criture se fait sur front descendant )
            ld pc,a
            nop
            nop
            clr pa               ; annule l'instruction ( histoire de ...)
            ldi wdr,chien
            ret



; Routine d'criture d'une donne  destination de l'afficheur LCD
; La donne  crire est contenu dans l'accu A
; ( initialement E = RW = RS = 0 , c'est l'tat par dfaut ! )

ecridata    ldi wdr,chien
            ld w,a               ; sauvegarde la donne  crire
ecr_4       call testbusy
            jrc ecr_4
            ld a,w               ; rcupre la donne
            ld pa,a              ; impose la donne  crire
            ld a,copypc          ; travaille sur image du port C
            set rs,a             ; impose RS = 1 ( tranfert de donnes )
            ld copypc,a
            ld pc,a
            nop
            nop
            ld a,copypc          ; travaille sur image du port C
            set e,a              ; impose  E = 1 pour valider l'criture
            ld copypc,a
            ld pc,a
            nop
            nop
            ld a,copypc          ; travaille sur image du port C
            res e,a              ; impose  E = 0 pour valider l'criture
            ld copypc,a          ; ( l'criture se fait sur front descendant )
            ld pc,a
            nop
            nop
            ld a,copypc          ; travaille sur image du port C
            res rs,a             ; impose  RS = 0
            ld copypc,a
            ld pc,a
            nop
            nop
            inc curseur          ; incrmente position curseur
            ld a,curseur         ; non, teste si position curseur >= 8
            cpi a,8
            jrnz ecr_5
            ldi a,80h+40h        ; oui, saut  la 2ime partie de l'cran
            jp ecrinstr          ; le RET se fera en fin de routine ECRINSTR
ecr_5       clr pa               ; raz port A, juste pour le fun ...
            ret

; routine d'mission d'une chaine de caractres en zone ROM
; ADRMESH  doit contenir les 6 bits de poids forts de l'adresse du message
; ADRMESL  doit contenir les 6 bits de poids faibles de l'adresse du message
; Le message doit se terminer par le caractre nul 00h
; Les registres utiliss sont : A , X
; L'appel est de la forme :  LDI ADRMESH, @.W
;                            LDI ADRMESL, @.D
;                            CALL PRINTMES

printmes    ld a,adrmesl
            ld x,a
pmes_1      ld a,adrmesh         ; recupre adresse du caractere courant
            ld drwr,a
pmes_2      ld a,(x)             ; recupre caractre courant du message
            jrnz pmes_0          ; si fin de message on rentre  la maison !
            ret
pmes_0      call ecridata        ; envoit le caractre sur afficheur LCD
            inc x                ; passe au caractre suivant
            jrr 7,x,pmes_2       ; teste si il faut incrmenter le pointeur de
            ldi x,40h            ; la fentre en ROM.
            inc adrmesh          ; Si oui, initialise  6 bits de poids failes
            jp pmes_1            ; et incrmente les 6 bits de poids forts.


;----------------------------------------------------------------------------
;                        Vecteurs d'interruption
;----------------------------------------------------------------------------

;  ***     CAN     ***
           .org 0FF0h
           nop
           reti

;  ***     TIMER   ***
           .ORG 0FF2h
           nop
           reti

;  ***     PORTB,C ***
           .org 0FF4h
           nop
           reti

;  ***     PORTA   ***
           .org 0FF6h
           nop
           reti

;  ***     NMI     ***
           .org 0FFCh
           nop
           reti

;  ***     RESET   ***
           .org 0FFEh
           jp StartProg

           .end
